/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.common;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Maps;
import java.lang.annotation.RetentionPolicy;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.apache.shindig.common.JsonAssert;
import org.apache.shindig.common.JsonProperty;
import org.apache.shindig.common.JsonSerializer;
import org.json.JSONArray;
import org.json.JSONObject;
import org.junit.Assert;
import org.junit.Test;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class JsonSerializerTest {
    private static final String JSON_POJO_AS_JSON = "{string:'string-value',integer:100,'simple!':3}";

    @Test
    public void serializeSimpleJsonObject() throws Exception {
        String json = "{foo:'bar'}";
        JsonAssert.assertJsonEquals(json, JsonSerializer.serialize((JSONObject)new JSONObject(json)));
    }

    @Test
    public void serializeSimpleMap() throws Exception {
        HashMap<String, String> map = new HashMap<String, String>(3, 1.0f);
        map.put("hello", "world");
        map.put("foo", "bar");
        map.put("remove", null);
        JsonAssert.assertJsonEquals("{hello:'world',foo:'bar'}", JsonSerializer.serialize(map));
    }

    @Test
    public void serializeSimpleMultimap() throws Exception {
        LinkedHashMultimap map = LinkedHashMultimap.create();
        ImmutableSet methods = ImmutableSet.of((Object)"system.listMethods", (Object)"people.get");
        map.putAll((Object)"hostEndpoint", (Iterable)methods);
        JsonAssert.assertJsonEquals("{hostEndpoint : ['system.listMethods', 'people.get']}", JsonSerializer.serialize((Object)map));
    }

    @Test
    public void serializeSimpleCollection() throws Exception {
        List<String> collection = Arrays.asList("foo", null, "bar", "baz", null);
        JsonAssert.assertJsonEquals("['foo','bar','baz']", JsonSerializer.serialize(collection));
    }

    @Test
    public void serializeArray() throws Exception {
        Object[] array = new String[]{"foo", null, "bar", "baz"};
        JsonAssert.assertJsonEquals("['foo','bar','baz']", JsonSerializer.serialize((Object[])array));
    }

    @Test
    public void serializeJsonArray() throws Exception {
        JSONArray array = new JSONArray((Object)new String[]{"foo", null, "bar", "baz"});
        JsonAssert.assertJsonEquals("['foo','bar','baz']", JsonSerializer.serialize((JSONArray)array));
    }

    @Test
    public void serializeJsonObjectWithComplexArray() throws Exception {
        JSONArray array = new JSONArray();
        array.put((Object)new JsonPojo());
        JSONObject object = new JSONObject();
        object.put("array", (Object)array);
        JsonAssert.assertJsonEquals("{'array': [{string:'string-value',integer:100,'simple!':3}]}", JsonSerializer.serialize((JSONObject)object));
    }

    @Test
    public void serializeJsonObjectWithNullPropertyValue() throws Exception {
        String json = "{foo:null}";
        JsonAssert.assertJsonEquals(json, JsonSerializer.serialize((JSONObject)new JSONObject(json)));
    }

    @Test
    public void serializePrimitives() throws Exception {
        Assert.assertEquals((Object)"null", (Object)JsonSerializer.serialize((Object)null));
        Assert.assertEquals((Object)"\"hello\"", (Object)JsonSerializer.serialize((Object)"hello"));
        Assert.assertEquals((Object)"100", (Object)JsonSerializer.serialize((Object)100));
        Assert.assertEquals((Object)"125.0", (Object)JsonSerializer.serialize((Object)Float.valueOf(125.0f)));
        Assert.assertEquals((Object)"126.0", (Object)JsonSerializer.serialize((Object)126.0));
        Assert.assertEquals((Object)"1", (Object)JsonSerializer.serialize((Object)1L));
        Assert.assertEquals((Object)"\"RUNTIME\"", (Object)JsonSerializer.serialize((Object)((Object)RetentionPolicy.RUNTIME)));
        Assert.assertEquals((Object)"\"string buf\"", (Object)JsonSerializer.serialize((Object)new StringBuilder().append("string").append(' ').append("buf")));
    }

    @Test
    public void serializePojo() throws Exception {
        JsonPojo pojo = new JsonPojo();
        JsonAssert.assertJsonEquals(JSON_POJO_AS_JSON, JsonSerializer.serialize((Object)pojo));
    }

    @Test
    public void serializeMixedObjects() throws Exception {
        ImmutableMap map = ImmutableMap.of((Object)"int", (Object)3, (Object)"double", (Object)2.7, (Object)"bool", (Object)Boolean.TRUE, (Object)"map", (Object)ImmutableMap.of((Object)"hello", (Object)"world", (Object)"foo", (Object)"bar"), (Object)"string", (Object)"hello!");
        JsonAssert.assertJsonEquals("{int:3,double:2.7,bool:true,map:{hello:'world',foo:'bar'},string:'hello!'}", JsonSerializer.serialize((Map)map));
    }

    @Test
    public void serializeMixedArray() throws Exception {
        List<Object> data = Arrays.asList(3, 2.7, Boolean.TRUE, Arrays.asList("one", "two", "three"), new JSONArray((Object)new String[]{"foo", "bar"}), "hello!");
        JsonAssert.assertJsonEquals("[3,2.7,true,['one','two','three'],['foo','bar'],'hello!']", JsonSerializer.serialize(data));
    }

    @Test
    public void emptyString() throws Exception {
        StringBuilder builder = new StringBuilder();
        JsonSerializer.appendString((Appendable)builder, (CharSequence)"");
        Assert.assertEquals((Object)"\"\"", (Object)builder.toString());
    }

    @Test
    public void escapeSequences() throws Exception {
        StringBuilder builder = new StringBuilder();
        JsonSerializer.appendString((Appendable)builder, (CharSequence)"\t\r value \\\foo\b\uffff\ubcad\n\u0083");
        Assert.assertEquals((Object)"\"\\t\\r value \\\\\\foo\\b\uffff\ubcad\\n\\u0083\"", (Object)builder.toString());
    }

    @Test
    public void escapeBrackets() throws Exception {
        StringBuilder builder = new StringBuilder();
        JsonSerializer.appendString((Appendable)builder, (CharSequence)"Hello<world>foo < bar");
        Assert.assertEquals((Object)"\"Hello\\u003cworld\\u003efoo \\u003c bar\"", (Object)builder.toString());
        JSONObject obj = new JSONObject("{foo:" + builder + '}');
        Assert.assertEquals((Object)"Hello<world>foo < bar", (Object)obj.get("foo"));
    }

    private static String avg(long start, long end, long runs) {
        double delta = end - start;
        return String.format("%f5", delta / (double)runs);
    }

    private static String runJsonOrgTest(Map<String, Object> data, int iterations) {
        JSONObject object = new JSONObject(data);
        long start = System.currentTimeMillis();
        String result = null;
        for (int i = 0; i < iterations; ++i) {
            result = object.toString();
        }
        System.out.println("json.org: " + JsonSerializerTest.avg(start, System.currentTimeMillis(), iterations) + "ms");
        return result;
    }

    private static String runSerializerTest(Map<String, Object> data, int iterations) {
        long start = System.currentTimeMillis();
        String result = null;
        for (int i = 0; i < iterations; ++i) {
            result = JsonSerializer.serialize(data);
        }
        System.out.println("serializer: " + JsonSerializerTest.avg(start, System.currentTimeMillis(), iterations) + "ms");
        return result;
    }

    public static Map<String, Object> perfComparison100SmallValues() {
        HashMap data = Maps.newHashMap();
        for (int i = 0; i < 100; ++i) {
            data.put("key-" + i, "small value");
        }
        return data;
    }

    public static Map<String, Object> perfComparison1000SmallValues() {
        HashMap data = Maps.newHashMap();
        for (int i = 0; i < 1000; ++i) {
            data.put("key-" + i, "small value");
        }
        return data;
    }

    public static Map<String, Object> perfComparison100LargeValues() {
        HashMap data = Maps.newHashMap();
        for (int i = 0; i < 100; ++i) {
            data.put("key-" + i, StringUtils.repeat((String)"small value", (int)100));
        }
        return data;
    }

    public static Map<String, Object> perfComparison10LargeValuesAndEscapes() {
        HashMap data = Maps.newHashMap();
        for (int i = 0; i < 10; ++i) {
            data.put("key-" + i, StringUtils.repeat((String)"\tsmall\r value \\foo\b\uffff\ubcad\n\u0083", (int)100));
        }
        return data;
    }

    public static Map<String, Object> perfComparison100Arrays() {
        HashMap data = Maps.newHashMap();
        String[] array = new String[]{"one", "two", "three", "four", "five", "six", "seven", "eight", "nine", "ten"};
        for (int i = 0; i < 100; ++i) {
            data.put("key-" + i, array);
        }
        return data;
    }

    public static void main(String[] args) throws Exception {
        int iterations = args.length > 0 ? Integer.parseInt(args[0]) : 1000;
        System.out.println("Running tests with " + iterations + " iterations.");
        for (Method method : JsonSerializerTest.class.getMethods()) {
            if (!method.getName().startsWith("perfComparison")) continue;
            Map data = (Map)method.invoke(null, new Object[0]);
            System.out.println("Running: " + method.getName());
            JsonSerializerTest.runJsonOrgTest(data, iterations);
            JsonSerializerTest.runSerializerTest(data, iterations);
            System.out.println("-----------------------");
        }
        System.out.println("Done");
    }

    public static class JsonPojo {
        public String getString() {
            return "string-value";
        }

        private String getPrivateString() {
            throw new UnsupportedOperationException();
        }

        public int getInteger() {
            return 100;
        }

        @JsonProperty(value="simple!")
        public int getSimpleName() {
            return 3;
        }

        public Object getNullValue() {
            return null;
        }

        @JsonProperty(value="simple!")
        public void setSimpleName(int foo) {
        }

        @JsonProperty(value="invalid-setter-two-args")
        public void setInvalidSetterTwoArgs(String foo, String bar) {
        }

        @JsonProperty(value="invalid-setter-no-args")
        public void setInvalidSetterNoArgs() {
        }

        @JsonProperty(value="invalid-getter-args")
        public String getInvalidGetterWithArgs(String foo) {
            return "invalid";
        }
    }
}

