/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.common.crypto;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.codec.binary.Base64;
import org.apache.shindig.common.crypto.BasicBlobCrypter;
import org.apache.shindig.common.crypto.BlobCrypterException;
import org.apache.shindig.common.crypto.BlobExpiredException;
import org.apache.shindig.common.crypto.Crypto;
import org.apache.shindig.common.util.FakeTimeSource;
import org.junit.Assert;
import org.junit.Test;

public class BlobCrypterTest {
    private BasicBlobCrypter crypter = new BasicBlobCrypter("0123456789abcdef".getBytes());
    private FakeTimeSource timeSource = new FakeTimeSource();

    public BlobCrypterTest() {
        this.crypter.timeSource = this.timeSource;
    }

    @Test
    public void testEncryptAndDecrypt() throws Exception {
        this.checkString("");
        this.checkString("a");
        this.checkString("ab");
        this.checkString("dfkljdasklsdfklasdjfklajsdfkljasdklfjasdkljfaskldjf");
        this.checkString(Crypto.getRandomString((int)500));
        this.checkString("foo bar baz");
        this.checkString("foo\nbar\nbaz");
    }

    private void checkString(String string) throws Exception {
        HashMap in = Maps.newHashMap();
        if (string != null) {
            in.put("a", string);
        }
        String blob = this.crypter.wrap((Map)in);
        Map out = this.crypter.unwrap(blob, 0);
        Assert.assertEquals((Object)string, out.get("a"));
    }

    @Test
    public void testDecryptGarbage() throws Exception {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < 100; ++i) {
            this.assertThrowsBlobCrypterException(sb.toString());
            sb.append('a');
        }
    }

    private void assertThrowsBlobCrypterException(String in) {
        try {
            this.crypter.unwrap(in, 1000);
            Assert.fail((String)("Should have thrown BlobCrypterException for input " + in));
        }
        catch (BlobCrypterException blobCrypterException) {
            // empty catch block
        }
    }

    @Test
    public void testManyEntries() throws Exception {
        HashMap in = Maps.newHashMap();
        for (int i = 0; i < 1000; ++i) {
            in.put(Integer.toString(i), Integer.toString(i));
        }
        String blob = this.crypter.wrap((Map)in);
        Map out = this.crypter.unwrap(blob, 0);
        for (int i = 0; i < 1000; ++i) {
            Assert.assertEquals(out.get(Integer.toString(i)), (Object)Integer.toString(i));
        }
    }

    @Test
    public void testTimeStamping() throws Exception {
        long start = 1201917724000L;
        long skew = 180000L;
        int maxAge = 300;
        int realAge = 600;
        try {
            this.timeSource.setCurrentTimeMillis(start);
            ImmutableMap in = ImmutableMap.of((Object)"a", (Object)"b");
            String blob = this.crypter.wrap((Map)in);
            this.timeSource.incrementSeconds(realAge);
            this.crypter.unwrap(blob, maxAge);
            Assert.fail((String)"Blob should have expired");
        }
        catch (BlobExpiredException e) {
            Assert.assertEquals((long)(start - skew), (long)e.minDate.getTime());
            Assert.assertEquals((long)(start + (long)realAge * 1000L), (long)e.used.getTime());
            Assert.assertEquals((long)(start + skew + (long)maxAge * 1000L), (long)e.maxDate.getTime());
        }
    }

    @Test(expected=BlobCrypterException.class)
    public void testTamperIV() throws Exception {
        ImmutableMap in = ImmutableMap.of((Object)"a", (Object)"b");
        String blob = this.crypter.wrap((Map)in);
        byte[] blobBytes = Base64.decodeBase64((byte[])blob.getBytes());
        blobBytes[0] = (byte)(blobBytes[0] ^ 1);
        String tampered = new String(Base64.encodeBase64((byte[])blobBytes));
        this.crypter.unwrap(tampered, 30);
    }

    @Test(expected=BlobCrypterException.class)
    public void testTamperData() throws Exception {
        ImmutableMap in = ImmutableMap.of((Object)"a", (Object)"b");
        String blob = this.crypter.wrap((Map)in);
        byte[] blobBytes = Base64.decodeBase64((byte[])blob.getBytes());
        blobBytes[30] = (byte)(blobBytes[30] ^ 1);
        String tampered = new String(Base64.encodeBase64((byte[])blobBytes));
        this.crypter.unwrap(tampered, 30);
    }

    @Test(expected=BlobCrypterException.class)
    public void testTamperMac() throws Exception {
        ImmutableMap in = ImmutableMap.of((Object)"a", (Object)"b");
        String blob = this.crypter.wrap((Map)in);
        byte[] blobBytes = Base64.decodeBase64((byte[])blob.getBytes());
        int n = blobBytes.length - 1;
        blobBytes[n] = (byte)(blobBytes[n] ^ 1);
        String tampered = new String(Base64.encodeBase64((byte[])blobBytes));
        this.crypter.unwrap(tampered, 30);
    }

    @Test
    public void testFixedKey() throws Exception {
        BasicBlobCrypter alt = new BasicBlobCrypter("0123456789abcdef".getBytes());
        ImmutableMap in = ImmutableMap.of((Object)"a", (Object)"b");
        String blob = this.crypter.wrap((Map)in);
        Map out = alt.unwrap(blob, 30);
        Assert.assertEquals((Object)"b", out.get("a"));
    }

    @Test(expected=BlobCrypterException.class)
    public void testBadKey() throws Exception {
        BasicBlobCrypter alt = new BasicBlobCrypter("1123456789abcdef".getBytes());
        ImmutableMap in = ImmutableMap.of((Object)"a", (Object)"b");
        String blob = this.crypter.wrap((Map)in);
        alt.unwrap(blob, 30);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testShortKeyFails() throws Exception {
        new BasicBlobCrypter("0123456789abcde".getBytes());
    }
}

