/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.http;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Arrays;
import org.apache.commons.io.IOUtils;
import org.apache.shindig.common.util.DateUtil;
import org.apache.shindig.common.util.FakeTimeSource;
import org.apache.shindig.common.util.TimeSource;
import org.apache.shindig.gadgets.http.HttpResponse;
import org.apache.shindig.gadgets.http.HttpResponseBuilder;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class HttpResponseTest
extends Assert {
    private static final byte[] UTF8_DATA = new byte[]{-17, -69, -65, 104, 101, 108, 108, 111};
    private static final String UTF8_STRING = "hello";
    private static final byte[] LATIN1_DATA = new byte[]{71, 97, 109, 101, 115, 44, 32, 72, 81, 44, 32, 77, 97, 110, 103, -31, 44, 32, 65, 110, 105, 109, 101, 32, 101, 32, 116, 117, 100, 111, 32, 113, 117, 101, 32, 117, 109, 32, 98, 111, 109, 32, 110, 101, 114, 100, 32, 97, 109, 97};
    private static final String LATIN1_STRING = "Games, HQ, Mang\u00e1, Anime e tudo que um bom nerd ama";
    private static final byte[] BIG5_DATA = new byte[]{-89, 65, -90, 110};
    private static final String BIG5_STRING = "\u4f60\u597d";
    private static FakeTimeSource timeSource = new FakeTimeSource(System.currentTimeMillis());

    private static int roundToSeconds(long ts) {
        return (int)(ts / 1000L);
    }

    @Before
    public void setUp() {
        HttpResponse.setTimeSource((TimeSource)timeSource);
    }

    @Test
    public void testEncodingDetectionUtf8WithBom() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=UTF-8").setResponse(UTF8_DATA).create();
        HttpResponseTest.assertEquals((Object)UTF8_STRING, (Object)response.getResponseAsString());
        HttpResponseTest.assertEquals((Object)"UTF-8", (Object)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomCaseInsensitiveKey() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; Charset=utf-8").setResponse(LATIN1_DATA).create();
        HttpResponseTest.assertEquals((Object)"UTF-8", (Object)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionLatin1() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=iso-8859-1").setResponse(LATIN1_DATA).create();
        HttpResponseTest.assertEquals((Object)LATIN1_STRING, (Object)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionLatin1withIncorrectCharset() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=iso-88859-1").setResponse(LATIN1_DATA).create();
        HttpResponseTest.assertEquals((Object)LATIN1_STRING, (Object)response.getResponseAsString());
        HttpResponseTest.assertEquals((Object)"ISO-8859-1", (Object)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomAndIncorrectCharset() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=UTTFF-88").setResponse(UTF8_DATA).create();
        HttpResponseTest.assertEquals((Object)UTF8_STRING, (Object)response.getResponseAsString());
        HttpResponseTest.assertEquals((Object)"UTF-8", (Object)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomAndInvalidCharset() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=.UTF-8").setResponse(UTF8_DATA).create();
        HttpResponseTest.assertEquals((Object)UTF8_STRING, (Object)response.getResponseAsString());
        HttpResponseTest.assertEquals((Object)"UTF-8", (Object)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionBig5() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=BIG5").setResponse(BIG5_DATA).create();
        HttpResponseTest.assertEquals((Object)BIG5_STRING, (Object)response.getResponseAsString());
        HttpResponseTest.assertEquals((Object)"text/plain; charset=BIG5", (Object)response.getHeader("Content-Type"));
        HttpResponseBuilder subResponseBuilder = new HttpResponseBuilder(response);
        subResponseBuilder.setContent(response.getResponseAsString());
        HttpResponse subResponse = subResponseBuilder.create();
        HttpResponseTest.assertEquals((Object)"text/plain; charset=UTF-8", (Object)subResponse.getHeader("Content-Type"));
        HttpResponseTest.assertEquals((Object)BIG5_STRING, (Object)subResponse.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionBig5WithQuotes() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=\"BIG5\"").setResponse(BIG5_DATA).create();
        HttpResponseTest.assertEquals((Object)BIG5_STRING, (Object)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomNoCharsetSpecified() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain").setResponse(UTF8_DATA).create();
        HttpResponseTest.assertEquals((Object)"UTF-8", (Object)response.getEncoding().toUpperCase());
        HttpResponseTest.assertEquals((Object)UTF8_STRING, (Object)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionLatin1NoCharsetSpecified() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain;").setResponse(LATIN1_DATA).create();
        HttpResponseTest.assertEquals((Object)"ISO-8859-1", (Object)response.getEncoding().toUpperCase());
        HttpResponseTest.assertEquals((Object)LATIN1_STRING, (Object)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionWithEmptyContentType() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "").setResponseString("something").create();
        HttpResponseTest.assertEquals((Object)HttpResponse.DEFAULT_ENCODING.name(), (Object)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomNoContentHeader() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(UTF8_DATA).create();
        HttpResponseTest.assertEquals((Object)"UTF-8", (Object)response.getEncoding().toUpperCase());
        HttpResponseTest.assertEquals((Object)UTF8_STRING, (Object)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionLatin1NoContentHeader() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(LATIN1_DATA).create();
        HttpResponseTest.assertEquals((Object)HttpResponse.DEFAULT_ENCODING.name(), (Object)response.getEncoding());
    }

    @Test
    public void testGetEncodingForImageContentType() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(LATIN1_DATA).addHeader("Content-Type", "image/png; charset=iso-8859-1").create();
        HttpResponseTest.assertEquals((Object)HttpResponse.DEFAULT_ENCODING.name(), (Object)response.getEncoding().toUpperCase());
    }

    @Test
    public void testGetEncodingForFlashContentType() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(LATIN1_DATA).addHeader("Content-Type", "application/x-shockwave-flash; charset=iso-8859-1").create();
        HttpResponseTest.assertEquals((Object)HttpResponse.DEFAULT_ENCODING.name(), (Object)response.getEncoding().toUpperCase());
    }

    @Test
    public void testPreserveBinaryData() throws Exception {
        byte[] data = new byte[]{0, -34, -22, -37, -18, -16};
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "application/octet-stream").setResponse(data).create();
        byte[] out = IOUtils.toByteArray((InputStream)response.getResponse());
        HttpResponseTest.assertEquals((long)data.length, (long)response.getContentLength());
        HttpResponseTest.assertTrue((boolean)Arrays.equals(data, out));
        out = IOUtils.toByteArray((InputStream)response.getResponse());
        HttpResponseTest.assertTrue((boolean)Arrays.equals(data, out));
    }

    @Test
    public void testStrictCacheControlNoCache() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").create();
        HttpResponseTest.assertTrue((boolean)response.isStrictNoCache());
        HttpResponseTest.assertEquals((long)-1L, (long)response.getCacheExpiration());
        HttpResponseTest.assertEquals((long)-1L, (long)response.getCacheTtl());
    }

    @Test
    public void testStrictPragmaNoCache() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Pragma", "no-cache").create();
        HttpResponseTest.assertTrue((boolean)response.isStrictNoCache());
        HttpResponseTest.assertEquals((long)-1L, (long)response.getCacheExpiration());
        HttpResponseTest.assertEquals((long)-1L, (long)response.getCacheTtl());
    }

    @Test
    public void testStrictPragmaJunk() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Pragma", "junk").create();
        HttpResponseTest.assertFalse((boolean)response.isStrictNoCache());
        int expected = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis() + response.getDefaultTtl());
        int expires = HttpResponseTest.roundToSeconds(response.getCacheExpiration());
        HttpResponseTest.assertEquals((long)expected, (long)expires);
        HttpResponseTest.assertTrue((response.getCacheTtl() <= response.getDefaultTtl() && response.getCacheTtl() > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testCachingHeadersIgnoredOnError() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").setHttpStatusCode(404).create();
        HttpResponseTest.assertFalse((boolean)response.isStrictNoCache());
        response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").setHttpStatusCode(403).create();
        HttpResponseTest.assertTrue((boolean)response.isStrictNoCache());
        response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").setHttpStatusCode(401).create();
        HttpResponseTest.assertTrue((boolean)response.isStrictNoCache());
    }

    private static void assertTtlOk(int maxAge, HttpResponse response) {
        HttpResponseTest.assertEquals((long)(maxAge - 1), (long)HttpResponseTest.roundToSeconds(response.getCacheTtl() - 1L));
    }

    @Test
    public void testExpires() throws Exception {
        int maxAge = 10;
        int time = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis()) + maxAge;
        HttpResponse response = new HttpResponseBuilder().addHeader("Expires", DateUtil.formatRfc1123Date((long)(1000L * (long)time))).create();
        HttpResponseTest.assertEquals((long)time, (long)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testExpiresZeroValue() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Expires", "0").create();
        HttpResponseTest.assertEquals((long)0L, (long)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
    }

    @Test
    public void testExpiresUnknownValue() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Expires", "howdy").create();
        HttpResponseTest.assertEquals((long)0L, (long)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
    }

    @Test
    public void testMaxAgeNoDate() throws Exception {
        int maxAge = 10;
        int expected = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis()) + maxAge;
        HttpResponse response = new HttpResponseBuilder().addHeader("Cache-Control", "public, max-age=" + maxAge).create();
        int expiration = HttpResponseTest.roundToSeconds(response.getCacheExpiration());
        HttpResponseTest.assertEquals((long)expected, (long)expiration);
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testMaxAgeInvalidDate() throws Exception {
        int maxAge = 10;
        int expected = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis()) + maxAge;
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", "Wed, 09 Jul 2008 19:18:33 EDT").addHeader("Cache-Control", "public, max-age=" + maxAge).create();
        int expiration = HttpResponseTest.roundToSeconds(response.getCacheExpiration());
        HttpResponseTest.assertEquals((long)expected, (long)expiration);
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testMaxAgeWithDate() throws Exception {
        int maxAge = 10;
        int now = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", DateUtil.formatRfc1123Date((long)(1000L * (long)now))).addHeader("Cache-Control", "public, max-age=" + maxAge).create();
        HttpResponseTest.assertEquals((long)(now + maxAge), (long)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testFixedDate() throws Exception {
        int time = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", DateUtil.formatRfc1123Date((long)(1000L * (long)time))).create();
        HttpResponseTest.assertEquals((long)(time + HttpResponseTest.roundToSeconds(response.getDefaultTtl())), (long)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertEquals((Object)DateUtil.formatRfc1123Date((long)timeSource.currentTimeMillis()), (Object)response.getHeader("Date"));
        HttpResponseTest.assertTtlOk(HttpResponseTest.roundToSeconds(response.getDefaultTtl()), response);
    }

    @Test
    public void testFixedDateOld() throws Exception {
        int time = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", DateUtil.formatRfc1123Date((long)(1000L * (long)time - 1000L - 180000L))).create();
        HttpResponseTest.assertEquals((long)(time + HttpResponseTest.roundToSeconds(response.getDefaultTtl())), (long)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertEquals((Object)DateUtil.formatRfc1123Date((long)timeSource.currentTimeMillis()), (Object)response.getHeader("Date"));
        HttpResponseTest.assertTtlOk(HttpResponseTest.roundToSeconds(response.getDefaultTtl()), response);
    }

    @Test
    public void testFixedDateNew() throws Exception {
        int time = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", DateUtil.formatRfc1123Date((long)(1000L * (long)time + 1000L + 180000L))).create();
        HttpResponseTest.assertEquals((long)(time + HttpResponseTest.roundToSeconds(response.getDefaultTtl())), (long)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertTtlOk(HttpResponseTest.roundToSeconds(response.getDefaultTtl()), response);
    }

    @Test
    public void testNegativeCaching() {
        HttpResponseTest.assertTrue((String)"Bad HTTP responses must be cacheable!", (HttpResponse.error().getCacheExpiration() > timeSource.currentTimeMillis() ? 1 : 0) != 0);
        HttpResponseTest.assertTrue((String)"Bad HTTP responses must be cacheable!", (HttpResponse.notFound().getCacheExpiration() > timeSource.currentTimeMillis() ? 1 : 0) != 0);
        HttpResponseTest.assertTrue((String)"Bad HTTP responses must be cacheable!", (HttpResponse.timeout().getCacheExpiration() > timeSource.currentTimeMillis() ? 1 : 0) != 0);
        long ttl = HttpResponse.error().getCacheTtl();
        HttpResponseTest.assertTrue((ttl <= 300000L && ttl > 0L ? 1 : 0) != 0);
    }

    private static void assertDoesNotAllowNegativeCaching(int status) {
        HttpResponse response = new HttpResponseBuilder().setHttpStatusCode(status).setResponse(UTF8_DATA).setStrictNoCache().create();
        HttpResponseTest.assertEquals((long)-1L, (long)response.getCacheTtl());
    }

    private static void assertAllowsNegativeCaching(int status) {
        HttpResponse response = new HttpResponseBuilder().setHttpStatusCode(status).setResponse(UTF8_DATA).setStrictNoCache().create();
        long ttl = response.getCacheTtl();
        HttpResponseTest.assertTrue((ttl <= response.getDefaultTtl() && ttl > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testStrictNoCacheAndNegativeCaching() {
        HttpResponseTest.assertDoesNotAllowNegativeCaching(401);
        HttpResponseTest.assertDoesNotAllowNegativeCaching(403);
        HttpResponseTest.assertDoesNotAllowNegativeCaching(200);
        HttpResponseTest.assertAllowsNegativeCaching(404);
        HttpResponseTest.assertAllowsNegativeCaching(500);
        HttpResponseTest.assertAllowsNegativeCaching(504);
    }

    @Test
    public void testRetryAfter() {
        for (int rc : Arrays.asList(500, 504, 400)) {
            HttpResponse response = new HttpResponseBuilder().setHttpStatusCode(rc).setHeader("Retry-After", "60").create();
            long ttl = response.getCacheTtl();
            HttpResponseTest.assertTrue((ttl <= 60000L && ttl > 0L ? 1 : 0) != 0);
        }
    }

    @Test
    public void testSetNoCache() {
        int time = HttpResponseTest.roundToSeconds(timeSource.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Expires", DateUtil.formatRfc1123Date((long)(1000L * (long)time))).setStrictNoCache().create();
        HttpResponseTest.assertNull((Object)response.getHeader("Expires"));
        HttpResponseTest.assertEquals((Object)"no-cache", (Object)response.getHeader("Pragma"));
        HttpResponseTest.assertEquals((Object)"no-cache", (Object)response.getHeader("Cache-Control"));
    }

    @Test
    public void testNullHeaderNamesStripped() {
        HttpResponse response = new HttpResponseBuilder().addHeader(null, "dummy").create();
        for (String key : response.getHeaders().keySet()) {
            HttpResponseTest.assertNotNull((String)"Null header not removed.", (Object)key);
        }
    }

    @Test
    public void testIsError() {
        HttpResponse response;
        int j;
        int i;
        for (i = 100; i < 400; i += 100) {
            for (j = 0; j < 10; ++j) {
                response = new HttpResponseBuilder().setHttpStatusCode(i).create();
                HttpResponseTest.assertFalse((String)"Status below 400 considered to be an error", (boolean)response.isError());
            }
        }
        for (i = 400; i < 600; i += 100) {
            for (j = 0; j < 10; ++j) {
                response = new HttpResponseBuilder().setHttpStatusCode(i).create();
                HttpResponseTest.assertTrue((String)"Status above 400 considered to be an error", (boolean)response.isError());
            }
        }
    }

    @Test
    public void testSerialization() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(baos);
        HttpResponse response = new HttpResponseBuilder().addHeader("Foo", "bar").addHeader("Foo", "baz").addHeader("Blah", "blah").setHttpStatusCode(204).setResponseString("This is the response string").create();
        out.writeObject(response);
        ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
        ObjectInputStream in = new ObjectInputStream(bais);
        HttpResponse deserialized = (HttpResponse)in.readObject();
        HttpResponseTest.assertEquals((Object)response, (Object)deserialized);
    }

    @Test
    public void testSerializationWithTransientFields() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(baos);
        long now = timeSource.currentTimeMillis();
        HttpResponse response = new HttpResponseBuilder().addHeader("Foo", "bar").addHeader("Foo", "baz").addHeader("Blah", "blah").addHeader("Date", DateUtil.formatRfc1123Date((long)now)).setHttpStatusCode(204).setResponseString("This is the response string").setMetadata("foo", "bar").create();
        out.writeObject(response);
        ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
        ObjectInputStream in = new ObjectInputStream(bais);
        HttpResponse deserialized = (HttpResponse)in.readObject();
        HttpResponse expectedResponse = new HttpResponseBuilder().addHeader("Foo", "bar").addHeader("Foo", "baz").addHeader("Blah", "blah").addHeader("Date", DateUtil.formatRfc1123Date((long)now)).setHttpStatusCode(204).setResponseString("This is the response string").create();
        HttpResponseTest.assertEquals((Object)expectedResponse, (Object)deserialized);
    }
}

