/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.shindig.social.opensocial.service;

import java.util.Set;
import java.util.concurrent.Future;

import org.apache.commons.lang.StringUtils;
import org.apache.shindig.protocol.HandlerPreconditions;
import org.apache.shindig.protocol.Operation;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.protocol.Service;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.GroupService;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.apache.shindig.social.opensocial.spi.VirtualOrganizationCollectionOptions;

import com.google.inject.Inject;

/**
 * RPC/REST handler for groups requests
 * 
 * @since 2.0.0
 */
@Service(name = "groups", path = "/{userId}")
public class GroupHandler {

  private final GroupService service;

  @Inject
  public GroupHandler(GroupService service) {
    this.service = service;
  }

  @Operation(httpMethods = "GET")
  public Future<?> get(SocialRequestItem request) throws ProtocolException {
    Set<UserId> userIds = request.getUsers();
    CollectionOptions options = getOptionsWithOptionalVoParam(request);

    // Preconditions
    HandlerPreconditions.requireNotEmpty(userIds, "No userId specified");
    HandlerPreconditions.requireSingular(userIds,
        "Only one userId must be specified");

    return service.getGroups(userIds.iterator().next(), options,
        request.getFields(), request.getToken());
  }

  private CollectionOptions getOptionsWithOptionalVoParam(
      SocialRequestItem request) {
    String vo = request.getParameter("vo");
    return StringUtils.isBlank(vo) ? new CollectionOptions(request)
        : new VirtualOrganizationCollectionOptions(request, vo);
  }

}
