/*
 * Decompiled with CFR 0.152.
 */
package oidc.secure;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.crypto.tink.Aead;
import com.google.crypto.tink.CleartextKeysetHandle;
import com.google.crypto.tink.JsonKeysetReader;
import com.google.crypto.tink.KeysetHandle;
import com.google.crypto.tink.KeysetReader;
import com.google.crypto.tink.aead.AeadConfig;
import com.google.crypto.tink.aead.AeadFactory;
import com.nimbusds.jose.Algorithm;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JOSEObjectType;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSHeader;
import com.nimbusds.jose.JWSSigner;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.RSASSASigner;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import com.nimbusds.oauth2.sdk.AuthorizationCode;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;
import com.nimbusds.openid.connect.sdk.Nonce;
import com.nimbusds.openid.connect.sdk.claims.AccessTokenHash;
import com.nimbusds.openid.connect.sdk.claims.CodeHash;
import com.nimbusds.openid.connect.sdk.claims.StateHash;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.text.ParseException;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import java.util.stream.Collectors;
import oidc.endpoints.MapTypeReference;
import oidc.exceptions.InvalidSignatureException;
import oidc.model.OpenIDClient;
import oidc.model.SigningKey;
import oidc.model.User;
import oidc.repository.SequenceRepository;
import oidc.repository.SigningKeyRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.env.Environment;
import org.springframework.core.env.Profiles;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

@Component
public class TokenGenerator
implements MapTypeReference {
    public static final JWSAlgorithm signingAlg = JWSAlgorithm.RS256;
    public static final Instant instant = Instant.parse("2100-01-01T00:00:00.00Z");
    private static char[] DEFAULT_CODEC = "1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".toCharArray();
    private Random random = new SecureRandom();
    private String issuer;
    private Map<String, JWSSigner> signers;
    private Map<String, JWSVerifier> verifiers;
    private String currentSigningKeyId;
    private List<RSAKey> publicKeys;
    private KeysetHandle keysetHandle;
    private String associatedData;
    private ObjectMapper objectMapper;
    private Clock clock;
    private SigningKeyRepository signingKeyRepository;
    private SequenceRepository sequenceRepository;

    @Autowired
    public TokenGenerator(@Value(value="${spring.security.saml2.service-provider.entity-id}") String issuer, @Value(value="${secret_key_set_path}") Resource secretKeySetPath, @Value(value="${associated_data}") String associatedData, ObjectMapper objectMapper, SigningKeyRepository signingKeyRepository, SequenceRepository sequenceRepository, Environment environment) throws IOException, GeneralSecurityException {
        AeadConfig.register();
        this.signingKeyRepository = signingKeyRepository;
        this.sequenceRepository = sequenceRepository;
        this.issuer = issuer;
        this.keysetHandle = CleartextKeysetHandle.read((KeysetReader)JsonKeysetReader.withInputStream((InputStream)secretKeySetPath.getInputStream()));
        this.associatedData = associatedData;
        this.objectMapper = objectMapper;
        this.clock = environment.acceptsProfiles(Profiles.of((String[])new String[]{"dev"})) ? Clock.fixed(instant, ZoneId.systemDefault()) : Clock.systemDefaultZone();
        this.initializeSigningKeys();
    }

    private void initializeSigningKeys() throws NoSuchProviderException, NoSuchAlgorithmException {
        List<Object> rsaKeys = this.signingKeyRepository.findAllByOrderByCreatedDesc().stream().map(arg_0 -> this.parseEncryptedRsaKey(arg_0)).collect(Collectors.toList());
        if (rsaKeys.isEmpty()) {
            SigningKey signingKey = this.generateEncryptedRsaKey();
            this.signingKeyRepository.save((Object)signingKey);
            RSAKey rsaKey = this.parseEncryptedRsaKey(signingKey);
            rsaKeys = Collections.singletonList(rsaKey);
        }
        this.publicKeys = rsaKeys.stream().map(RSAKey::toPublicJWK).collect(Collectors.toList());
        this.currentSigningKeyId = ((RSAKey)rsaKeys.get(0)).getKeyID();
        this.signers = rsaKeys.stream().collect(Collectors.toMap(JWK::getKeyID, arg_0 -> this.createRSASigner(arg_0)));
        this.verifiers = rsaKeys.stream().collect(Collectors.toMap(JWK::getKeyID, arg_0 -> this.createRSAVerifier(arg_0)));
    }

    public SigningKey rolloverSigningKeys() throws NoSuchProviderException, NoSuchAlgorithmException {
        SigningKey signingKey = this.generateEncryptedRsaKey();
        this.signingKeyRepository.save((Object)signingKey);
        this.initializeSigningKeys();
        return signingKey;
    }

    private RSAKey parseEncryptedRsaKey(SigningKey signingKey) {
        try {
            return RSAKey.parse((String)this.decryptAead(signingKey.getJwk()));
        }
        catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    public String generateAccessToken() {
        return UUID.randomUUID().toString();
    }

    public String generateRefreshToken() {
        return UUID.randomUUID().toString();
    }

    public String generateAuthorizationCode() {
        byte[] verifierBytes = new byte[12];
        this.random.nextBytes(verifierBytes);
        char[] chars = new char[verifierBytes.length];
        for (int i = 0; i < verifierBytes.length; ++i) {
            chars[i] = DEFAULT_CODEC[(verifierBytes[i] & 0xFF) % DEFAULT_CODEC.length];
        }
        return new String(chars);
    }

    public String generateAccessTokenWithEmbeddedUserInfo(User user, OpenIDClient client, List<String> scopes) {
        try {
            return this.doGenerateAccessTokenWithEmbeddedUser(user, client, scopes);
        }
        catch (Exception e) {
            throw e instanceof RuntimeException ? (RuntimeException)e : new RuntimeException(e);
        }
    }

    private String doGenerateAccessTokenWithEmbeddedUser(User user, OpenIDClient client, List<String> scopes) throws JsonProcessingException, GeneralSecurityException, JOSEException {
        String json = this.objectMapper.writeValueAsString((Object)user);
        String encryptedClaims = this.encryptAead(json);
        HashMap<String, String> additionalClaims = new HashMap<String, String>();
        additionalClaims.put("claims", encryptedClaims);
        additionalClaims.put("claim_key_id", this.currentSigningKeyId);
        return this.idToken(client, Optional.empty(), additionalClaims, Collections.emptyList(), true);
    }

    private String encryptAead(String s) {
        try {
            Aead aead = AeadFactory.getPrimitive((KeysetHandle)this.keysetHandle);
            byte[] src = aead.encrypt(s.getBytes(Charset.defaultCharset()), this.associatedData.getBytes(Charset.defaultCharset()));
            return Base64.getEncoder().encodeToString(src);
        }
        catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    public User decryptAccessTokenWithEmbeddedUserInfo(String accessToken) {
        try {
            return this.doDecryptAccessTokenWithEmbeddedUserInfo(accessToken);
        }
        catch (Exception e) {
            throw e instanceof RuntimeException ? (RuntimeException)e : new RuntimeException(e);
        }
    }

    private User doDecryptAccessTokenWithEmbeddedUserInfo(String accessToken) throws ParseException, JOSEException, GeneralSecurityException, IOException {
        SignedJWT signedJWT = SignedJWT.parse((String)accessToken);
        Map claims = this.verifyClaims(signedJWT);
        String encryptedClaims = (String)claims.get("claims");
        String s = this.decryptAead(encryptedClaims);
        return (User)this.objectMapper.readValue(s, User.class);
    }

    private String decryptAead(String s) {
        try {
            Aead aead = AeadFactory.getPrimitive((KeysetHandle)this.keysetHandle);
            byte[] decoded = Base64.getDecoder().decode(s);
            return new String(aead.decrypt(decoded, this.associatedData.getBytes(Charset.defaultCharset())));
        }
        catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    public String generateIDTokenForTokenEndpoint(Optional<User> user, OpenIDClient client, String nonce, List<String> idTokenClaims) throws JOSEException, NoSuchProviderException, NoSuchAlgorithmException {
        Map additionalClaims = StringUtils.hasText((String)nonce) ? Collections.singletonMap("nonce", nonce) : Collections.emptyMap();
        return this.idToken(client, user, additionalClaims, idTokenClaims, false);
    }

    public String generateIDTokenForAuthorizationEndpoint(User user, OpenIDClient client, Nonce nonce, ResponseType responseType, String accessToken, List<String> claims, Optional<String> authorizationCode, State state) throws JOSEException, NoSuchProviderException, NoSuchAlgorithmException {
        HashMap<String, String> additionalClaims = new HashMap<String, String>();
        if (nonce != null) {
            additionalClaims.put("nonce", nonce.getValue());
        }
        if (AccessTokenHash.isRequiredInIDTokenClaims((ResponseType)responseType)) {
            additionalClaims.put("at_hash", AccessTokenHash.compute((AccessToken)new BearerAccessToken(accessToken), (JWSAlgorithm)signingAlg).getValue());
        }
        if (CodeHash.isRequiredInIDTokenClaims((ResponseType)responseType) && authorizationCode.isPresent()) {
            additionalClaims.put("c_hash", (String)CodeHash.compute((AuthorizationCode)new AuthorizationCode(authorizationCode.get()), (JWSAlgorithm)signingAlg));
        }
        if (state != null && StringUtils.hasText((String)state.getValue())) {
            additionalClaims.put("s_hash", (String)StateHash.compute((State)state, (JWSAlgorithm)signingAlg));
        }
        return this.idToken(client, Optional.of(user), additionalClaims, claims, false);
    }

    public List<JWK> getAllPublicKeys() {
        return new ArrayList<JWK>(this.publicKeys);
    }

    public RSAKey generateRsaKey(String keyID) throws NoSuchAlgorithmException, NoSuchProviderException {
        KeyPairGenerator kpg = KeyPairGenerator.getInstance("RSA", "BC");
        kpg.initialize(2048);
        KeyPair keyPair = kpg.generateKeyPair();
        RSAPrivateKey privateKey = (RSAPrivateKey)keyPair.getPrivate();
        RSAPublicKey publicKey = (RSAPublicKey)keyPair.getPublic();
        return new RSAKey.Builder(publicKey).privateKey(privateKey).algorithm(new Algorithm("RS256")).keyID(keyID).build();
    }

    private SigningKey generateEncryptedRsaKey() throws NoSuchProviderException, NoSuchAlgorithmException {
        Long increment = this.sequenceRepository.increment();
        RSAKey rsaKey = this.generateRsaKey(this.signingKeyFormat(increment));
        String encryptedKey = this.encryptAead(rsaKey.toJSONString());
        return new SigningKey(rsaKey.getKeyID(), encryptedKey, new Date());
    }

    private String signingKeyFormat(Long increment) {
        return String.format("key_%s", increment);
    }

    private Map<String, Object> verifyClaims(SignedJWT signedJWT) throws ParseException, JOSEException, NoSuchProviderException, NoSuchAlgorithmException {
        this.ensureLatestSigningKey();
        String keyID = signedJWT.getHeader().getKeyID();
        if (!signedJWT.verify((JWSVerifier)this.verifiers.getOrDefault(keyID, this.verifiers.values().iterator().next()))) {
            throw new InvalidSignatureException("Tampered JWT");
        }
        return signedJWT.getJWTClaimsSet().getClaims();
    }

    private String idToken(OpenIDClient client, Optional<User> user, Map<String, Object> additionalClaims, List<String> idTokenClaims, boolean includeAllowedResourceServers) throws JOSEException, NoSuchProviderException, NoSuchAlgorithmException {
        ArrayList<String> audiences = new ArrayList<String>();
        audiences.add(client.getClientId());
        if (includeAllowedResourceServers) {
            audiences.addAll(client.getAllowedResourceServers().stream().filter(rsEntityId -> !client.getClientId().equals(rsEntityId)).collect(Collectors.toList()));
        }
        JWTClaimsSet.Builder builder = new JWTClaimsSet.Builder().audience(audiences).expirationTime(Date.from(this.clock.instant().plus((long)client.getAccessTokenValidity(), ChronoUnit.SECONDS))).jwtID(UUID.randomUUID().toString()).issuer(this.issuer).issueTime(Date.from(this.clock.instant())).subject(user.map(u -> u.getSub()).orElse(client.getClientId())).notBeforeTime(new Date(System.currentTimeMillis()));
        if (!CollectionUtils.isEmpty(idTokenClaims) && user.isPresent()) {
            Map attributes = user.get().getAttributes();
            idTokenClaims.forEach(claim -> {
                if (attributes.containsKey(claim)) {
                    builder.claim(claim, attributes.get(claim));
                }
            });
        }
        additionalClaims.forEach((name, value) -> builder.claim(name, value));
        JWTClaimsSet claimsSet = builder.build();
        JWSHeader header = new JWSHeader.Builder(signingAlg).type(JOSEObjectType.JWT).keyID(this.currentSigningKeyId).build();
        SignedJWT signedJWT = new SignedJWT(header, claimsSet);
        this.ensureLatestSigningKey();
        signedJWT.sign((JWSSigner)this.signers.getOrDefault(signedJWT.getHeader().getKeyID(), this.signers.values().iterator().next()));
        return signedJWT.serialize();
    }

    private void ensureLatestSigningKey() throws NoSuchProviderException, NoSuchAlgorithmException {
        if (!this.signingKeyFormat(this.sequenceRepository.currentSequence()).equals(this.currentSigningKeyId)) {
            this.initializeSigningKeys();
        }
    }

    private RSASSASigner createRSASigner(RSAKey k) {
        try {
            return new RSASSASigner(k);
        }
        catch (JOSEException e) {
            throw new RuntimeException(e);
        }
    }

    private RSASSAVerifier createRSAVerifier(RSAKey k) {
        try {
            return new RSASSAVerifier(k);
        }
        catch (JOSEException e) {
            throw new RuntimeException(e);
        }
    }

    public String getCurrentSigningKeyId() {
        return this.currentSigningKeyId;
    }
}

