/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.control;

import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.GadgetDefinition;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.domain.UserPreferences;
import nl.surfnet.coin.portal.service.GadgetDefinitionService;
import nl.surfnet.coin.portal.service.MetadataProvider;
import nl.surfnet.coin.portal.service.impl.GadgetDefinitionBuilderImpl;
import nl.surfnet.coin.portal.util.CoinEnvironment;
import nl.surfnet.coin.shared.service.SortOrder;
import org.apache.log4j.Logger;
import org.opensocial.models.Person;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletRequest;

import java.util.List;

/**
 * {@link Controller} that handles the home page of a logged in user.
 */
@Controller
public class GadgetController extends BaseController {
  private Logger logger = Logger.getLogger(this.getClass());

  @Autowired
  private MetadataProvider metadataProvider;

  @Autowired
  private CoinEnvironment environment;

  @Autowired
  private GadgetDefinitionService gadgetDefinitionService;

  private static final String ERROR_RESPONSE = "error";

  @RequestMapping("/gadgetoverview.shtml")
  public String gadgetOverview(ModelMap modelMap) {
    modelMap.addAttribute("environment", environment);

    // Get all gadgets
    List<GadgetDefinition> allGadgets =
        gadgetDefinitionService.findAllPublishedExcludeCustomOrderByPopularity(SortOrder.DESCENDING);

    modelMap.addAttribute("gadgetList", allGadgets);

    return "gadgetoverview";
  }

  @RequestMapping(value = "/search-gadget.shtml", method = RequestMethod.POST)
  public String searchGadget(ModelMap modelMap, HttpServletRequest request) {
    modelMap.addAttribute("environment", environment);

    // Get all gadgets
    String query = request.getParameter("gadgetQuery");

    // Check if the string isn't empty
    if (!StringUtils.hasText(query)) {
      query = "%";
    }

    // Check if the string ends with the wildcard
    if (!StringUtils.endsWithIgnoreCase(query, "%")) {
      query = query + "%";
    }

    // Check if the string starts with the wildcard
    if (!StringUtils.startsWithIgnoreCase(query, "%")) {
      query = "%" + query;
    }

    List<GadgetDefinition> gadgetDefinitions = gadgetDefinitionService
        .searchByWildCardExcludeCustom(query);

    modelMap.addAttribute("gadgetList", gadgetDefinitions);
    modelMap.addAttribute("search", true);
    modelMap.addAttribute("gadgetQuery", request.getParameter("gadgetQuery"));

    return "gadgetoverview";
  }

  @RequestMapping(value = "add-gadget-to-tab.shtml", method = RequestMethod.POST)
  public
  @ResponseBody
  String addGadget(HttpServletRequest request) {

    final UserPreferences userPreferences = getPreferences(request);
    if (userPreferences != null) {
      String alwaysGrant = request.getParameter("alwaysGrant");
      if (alwaysGrant == null || "false".equals(alwaysGrant)) {
        userPreferences.setInformationSharingPermissionGranted(false);
      } else {
        userPreferences.setInformationSharingPermissionGranted(true);
      }
      userPreferencesService.saveOrUpdate(userPreferences);
    }

    Tab tab = getTab(request, request.getParameter("tab"));
    if (tab == null) {
      logger.info("Unable to resolve the tab");
      return ERROR_RESPONSE;
    }

    String gadgetId = request.getParameter("gadget");

    if (!StringUtils.hasText(gadgetId)) {
      logger.info("Not a well formed gadget-id");
      return ERROR_RESPONSE;
    }

    GadgetDefinition gadgetDefinition = gadgetDefinitionService
          .findById(new Long(gadgetId));
    return addGadgetAndReturnTab(request, tab, gadgetDefinition);
  }

  @RequestMapping(value = "reorder-gadgets.shtml", method = RequestMethod.POST)
  public
  @ResponseBody
  boolean reorder(HttpServletRequest request) {
    String[] column1 = request.getParameterValues("gadgetIdsColumn1[]");
    String[] column2 = request.getParameterValues("gadgetIdsColumn2[]");
    Tab tab = getTab(request, request.getParameter("tabId"));
    if (tab == null) {
      logger.info("Could not find the tabId");
      return false;
    }
    return gadgetService.reorderGadgets(column1, column2, tab);
  }

  @RequestMapping(value = "delete-gadget.shtml", method = RequestMethod.POST)
  public
  @ResponseBody
  boolean delete(HttpServletRequest request) {
    Gadget gadget = getGadget(request, request.getParameter("gadget"));
    Tab tab = gadget.getTab();
    Person owner = getPerson(request);
    if (tab.getOwner().equals(owner.getId())) {
      gadgetService.delete(gadget);
      return true;
    }
    return false;
  }

  /**
   * If custom gadgets are allowed it creates a new {@link GadgetDefinition} or use an
   * existing based on its url and adds a {@link Gadget} to the requested {@link} Tab.
   *
   * @param request POST request
   * @return either the identifier of the Tab or {@value #ERROR_RESPONSE} if anything goes wrong.
   */
  @RequestMapping(value = "add-custom-gadget.shtml", method = RequestMethod.POST)
  public
  @ResponseBody
  String addCustomGadget(HttpServletRequest request) {

    // is it allowed to add a custom gadget?
    if (!environment.isAllowCustomGadgets()) {
      logger.warn("Custom gadgets are not allowed but someone is trying to add one");
      return ERROR_RESPONSE;
    }

    Tab tab = getTab(request, request.getParameter("tab"));
    if (tab == null) {
      logger.warn("Unable to resolve the tab");
      return ERROR_RESPONSE;
    }

    String gadgetUrl = request.getParameter("gadgetUrl");

    if (!StringUtils.hasText(gadgetUrl)) {
      logger.debug("GadgetUrl is empty");
      return ERROR_RESPONSE;
    }

    GadgetDefinition gadgetDefinition = gadgetDefinitionService.searchByUrl(gadgetUrl);

    if (gadgetDefinition == null) {
      gadgetDefinition = new GadgetDefinitionBuilderImpl(metadataProvider).build(gadgetUrl);
    }
    
    if (gadgetDefinition == null) {
      return ERROR_RESPONSE;
    }
    gadgetDefinitionService.saveOrUpdate(gadgetDefinition);
    return addGadgetAndReturnTab(request, tab, gadgetDefinition);

  }

  /**
   * Adds a {@link Gadget} to a {@link Tab} and returns the id of the Tab or an error code
   * in case something goes wrong.
   *
   * @param request          current {@link HttpServletRequest}
   * @param tab              The Tab to add the Gadget to
   * @param gadgetDefinition {@link GadgetDefinition} for the Gadget to add
   * @return either the identifier of the Tab or {@value #ERROR_RESPONSE} in case
   *         something goes wrong
   */
  private String addGadgetAndReturnTab(HttpServletRequest request,
                                       Tab tab,
                                       GadgetDefinition gadgetDefinition) {

    Gadget gadget = new Gadget();
    gadget.setHasPermission(true);
    gadget.setDefinition(gadgetDefinition);
    gadget.setTeam(tab.getTeam());

    // First move all the gadgets one spot downwards (increase order by one)
    gadgetService.clearFirstSpot(tab);

    // Add new Gadget on left top of page
    gadget.setOrder(0);
    gadget.setColumn(1);

    tab.addGadget(gadget);

    tabService.saveOrUpdate(tab);

    gadgetDefinition.setInstallCount(gadgetDefinition.getInstallCount() + 1);

    gadgetDefinitionService.saveOrUpdate(gadgetDefinition);

    return tab.getId().toString();
  }

  public void setEnvironment(CoinEnvironment environment) {
    this.environment = environment;
  }

  public void setMetadataProvider(MetadataProvider metadataProvider) {
    this.metadataProvider = metadataProvider;
  }
}
