/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.control;

import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.Invite;
import nl.surfnet.coin.portal.domain.InviteStatus;
import nl.surfnet.coin.portal.domain.SharedResource;
import nl.surfnet.coin.portal.domain.SharedResourceType;
import nl.surfnet.coin.portal.domain.SharedTab;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.service.InviteService;
import nl.surfnet.coin.portal.util.GroupList;
import nl.surfnet.coin.shindig.spi.CoinActivityService;
import org.apache.commons.lang.ArrayUtils;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.social.core.model.GroupImpl;
import org.apache.shindig.social.opensocial.model.Activity;
import org.apache.shindig.social.opensocial.model.Group;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.GroupId.Type;
import org.opensocial.models.Person;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.view.RedirectView;

import javax.servlet.http.HttpServletRequest;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;

/**
 * Controller for invites (SharedTabs and activities)
 */
@Controller
public class InvitesController extends BaseController {

  @Autowired
  private CoinActivityService activityService;

  @Autowired
  private InviteService inviteService;

  @RequestMapping("/invites.shtml")
  public String invites(ModelMap modelMap, HttpServletRequest request)
          throws ProtocolException, InterruptedException, ExecutionException {
    Person owner = getPerson(request);
    addInvites(modelMap, owner, request);
    GroupList groupList = getGroupsForLoggedInPerson(request);
    List<Group> shindigGroups = groupList.getShindigGroups();
    addActivities(modelMap, request, shindigGroups);
    return "invites";
  }

  @RequestMapping("/show-detail-invite.shtml")
  public String showInviteDetail(ModelMap modelMap, HttpServletRequest request) {
    String id = request.getParameter("id");
    Invite invite = inviteService.findById(Long.parseLong(id));
    SharedResource sharedResource = invite.getSharedResource();
    modelMap.addAttribute("invite", invite);
    if (sharedResource.getType().equals(SharedResourceType.Tab)) {
      modelMap.addAttribute("sharedTab", sharedResource);
      int columnSwitch = (((SharedTab) sharedResource).getPrototype()
              .getGadgets().size() / 2);
      modelMap.addAttribute("columnSwitch", ++columnSwitch);
      return "invite-detail-tab";
    }
    modelMap.addAttribute("sharedGadget", sharedResource);
    return "invite-detail-gadget";
  }

  @RequestMapping("/more-activities.shtml")
  public String moreActivities(ModelMap modelMap, HttpServletRequest request) {
    String groupId = request.getParameter("group_id");
    Group shindigGroup = new GroupImpl();
    shindigGroup.setId(new GroupId(Type.groupId, groupId));

    addActivities(modelMap, request, Collections.singletonList(shindigGroup));
    return "activities";
  }

  /*
   * Add the invites to the modelMap
   */
  private void addInvites(ModelMap modelMap, Person owner,
                          HttpServletRequest request) throws ProtocolException,
          InterruptedException, ExecutionException {
    InviteStatus status = getRequestedInviteStatus(request);
    List<Invite> invites = inviteService.findRecentByInvitee(owner.getId(), status);
    modelMap.addAttribute("invites", invites);
  }

  /*
   * Add activities
   */
  private void addActivities(ModelMap modelMap, HttpServletRequest request,
                             List<Group> groups) {
    String max = request.getParameter("max");
    int maxResult = (StringUtils.hasText(max) ? Integer.parseInt(max) : 100);
    maxResult++; // Get one more to check if there should be a More button
    String first = request.getParameter("first");
    int firstResult = (StringUtils.hasText(first) ? Integer.parseInt(first) : 0);

    Map<Group, List<Activity>> groupedActivities = new HashMap<Group, List<Activity>>();
    for (Group group : groups) {
      List<Activity> activitiesByGroup = activityService.findActivityByGroups(
              group, maxResult, firstResult);
      if (!CollectionUtils.isEmpty(activitiesByGroup)) {
        groupedActivities.put(group, activitiesByGroup);
      }
    }
    //modelMap.addAttribute("counter", first);
    modelMap.addAttribute("maxResult", maxResult - 1);
    modelMap.addAttribute("groupedActivities", groupedActivities);
  }

  /*
   * See which Invites is requested
   */
  private InviteStatus getRequestedInviteStatus(HttpServletRequest request) {
    String statusString = request.getParameter("status");
    return (StringUtils.hasText(statusString)) ? InviteStatus
            .valueOf(statusString.toUpperCase()) : InviteStatus.OPEN;
  }

  @RequestMapping("/deny-shared-tab-invite.shtml")
  public
  @ResponseBody
  boolean deny(ModelMap modelMap, HttpServletRequest request) {
    String id = request.getParameter("id");
    long idLong = Long.parseLong(id);
    Invite invite = inviteService.findById(idLong);
    Person owner = getPerson(request);
    if (!invite.getInvitee().equals(owner.getId())) {
      throw new IllegalArgumentException("Invite is not owned by '"
              + owner.getDisplayName() + "'");
    }
    invite.setStatus(InviteStatus.DENIED);
    inviteService.saveOrUpdate(invite);
    return true;

  }

  @RequestMapping(value = "/accept-invite.shtml", method = RequestMethod.POST)
  public View accept(ModelMap modelMap, HttpServletRequest request) {
    String inviteId = request.getParameter("inviteId");
    String[] gadgetIds = request.getParameterValues("add-gadget");
    String[] shareInfo = request.getParameterValues("share-info");

    // Get owner
    Person owner = getPerson(request);

    Invite invite = inviteService.findById(Long.parseLong(inviteId));
    SharedResource sharedResource = invite.getSharedResource();
    if (!invite.getInvitee().equals(owner.getId())) {
      throw new IllegalArgumentException("Invite is not owned by '"
              + owner.getDisplayName() + "'");
    }

    // Create tab
    Tab tab = new Tab();
    tab.setName(sharedResource.getName());
    tab.setFavorite(true);
    tab.setOrder(tabService.getCountByOwner(owner.getId(), true) + 1);
    tab.setOwner(owner.getId());
    tab.setTeam(sharedResource.getTeam());
    if (!ArrayUtils.isEmpty(gadgetIds)) {
      for (String gadgetId : gadgetIds) {
        // Only add gadgets that have permission to add a users' personal
        // information
        if (hasPermission(shareInfo, gadgetId)) {
          Gadget sharedGadget = gadgetService
                  .findById(Long.parseLong(gadgetId));
          Gadget gadget = new Gadget();
          gadget.setColumn(sharedGadget.getColumn());
          gadget.setOrder(sharedGadget.getOrder());
          gadget.setDefinition(sharedGadget.getDefinition());
          gadget.setHasPermission(true);
          gadget.setPrefs(sharedGadget.getPrefs());
          tab.addGadget(gadget);
        }
      }
    }

    // Save tab
    Long tabId = tabService.saveOrUpdate(tab);

    // Change Status invite to Accepted
    invite.setStatus(InviteStatus.ACCEPTED);
    inviteService.saveOrUpdate(invite);

    // return user to new tab
    return new RedirectView("home.shtml?tab=" + tabId.toString());
  }

  private boolean hasPermission(String[] shareInfo, String gadgetId) {
    if (shareInfo == null) {
      return false;
    }
    for (String shareInfoId : shareInfo) {
      if (shareInfoId.equals(gadgetId)) {
        return true;
      }
    }
    return false;
  }

}