/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.control;

import nl.surfnet.coin.portal.domain.ClonedTab;
import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.Invite;
import nl.surfnet.coin.portal.domain.InviteStatus;
import nl.surfnet.coin.portal.domain.SharedResource;
import nl.surfnet.coin.portal.domain.SharedTab;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.interceptor.LoginInterceptor;
import nl.surfnet.coin.portal.service.SharedResourceService;
import nl.surfnet.coin.portal.util.CoinEnvironment;
import nl.surfnet.coin.portal.util.GroupList;
import nl.surfnet.coin.shared.service.MailService;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.social.core.model.ActivityImpl;
import org.apache.shindig.social.opensocial.model.Activity;
import org.apache.shindig.social.opensocial.spi.ActivityService;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.opensocial.models.Group;
import org.opensocial.models.Person;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.mail.MailException;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.LocaleResolver;

import javax.servlet.http.HttpServletRequest;

import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.ExecutionException;

/**
 * {@link Controller} that handles the Team settings
 */
@Controller
public class TeamController extends BaseController {

  private static final String INVITE_FROM_ADRESS = "noreply@surfnet.nl";

  private static final String INVITE_MESSAGE_SUBJECT = "[SURFconext Portal] You have been invited";

  private static final String APP_ID_INVITE = "Portal";

  private static final String INVITE_ACTIVITY = "{0} has been shared.";

  // EN and NL are both together in one template atm:
  private static final String TEMPLATE_INVITES_EMAIL_PLAIN_EN = "template-invites-email_en_nl.plain";
  private static final String TEMPLATE_INVITES_EMAIL_PLAIN_NL = "template-invites-email_en_nl.plain";
  public static final String ERROR = "error";
  private static final String SUCCES = "succes";

  private static Logger logger = LoggerFactory.getLogger(TeamController.class);

  @Autowired
  private CoinEnvironment environment;

  @Autowired
  private SharedResourceService sharedResourceService;

  @Autowired
  private ActivityService activityService;

  @Autowired
  private LocaleResolver localeResolver;

  @Autowired
  private MailService mailService;

  @RequestMapping("/slidedown/team-settings.shtml")
  public String gadgetOverview(ModelMap modelMap, HttpServletRequest request)
          throws InterruptedException, ExecutionException {

    modelMap.addAttribute("environment", environment);

    final String tabId = request.getParameter("tab");
    Tab tab = getTab(request, tabId);
    if (tab == null) {
      throw new TabNotFoundException("Could not find tab with id " + tabId);
    }

    modelMap.addAttribute("requestedTab", tab);
    modelMap.addAttribute("gadgets", tab.getGadgets());

    // Get the groupList containing the users' groups
    GroupList groupList = getGroupsForLoggedInPerson(request);
    List<Group> groups = groupList.getGroups();

    Group[] groupsArray = groups.toArray(new Group[]{ });
    modelMap.addAttribute("teams", groupsArray);

    // Get the Group to retrieve the Name of the group
    if (StringUtils.hasText(tab.getTeam())) {

      Group group = groupList.getGroupById(tab.getTeam());
      modelMap.addAttribute("team", group);
      // Get the team members and add them to the modelMap
      // Get the owner
      Person owner = getPerson(request);

      List<Person> members = personService.getPeople(group.getId(), owner.getId());
      // Remove the owner of the list, so he cannot be invited
      // standard remove method of List does not work...
      members = removeOwner(members, owner.getId());
      members.remove(owner);

      // Place the invitees on the session in order to retrieve them when a
      // snapshot is send.
      request.getSession().setAttribute(LoginInterceptor.INVITEES, members);

      modelMap.addAttribute("members", members);
    }

    return "slidedown/team-settings";
  }

  @RequestMapping(value = "send-snapshot.shtml", method = RequestMethod.POST)
  public
  @ResponseBody
  String sendSnapshot(ModelMap modelMap, HttpServletRequest request)
          throws InterruptedException, ExecutionException, MailException, ProtocolException, IOException {
    String tabId = request.getParameter("id");
    String groupId = request.getParameter("team");
    String message = request.getParameter("message");
    String[] invitees = request.getParameterValues("invitees[]");
    String[] gadgets = request.getParameterValues("gadgets[]");

    // Tab, team or invitees are empty
    if (!StringUtils.hasText(tabId) || !StringUtils.hasText(groupId)
            || ArrayUtils.isEmpty(invitees)) {
      logger.warn("The required parameters are not set");
      return ERROR;
    }

    // Get Locale
    Locale locale = localeResolver.resolveLocale(request);

    // Get the owner
    Person owner = getPerson(request);

    // is this person a member of the group?
    if (!isMemberOfGroup(request, groupId)) {
      logger.warn("Someone tried to send a snapshot to a group, of which this user is not a member");
      return ERROR;
    }

    // Get the tab
    Tab tab = getTab(request, tabId);
    if (tab == null) {
      logger.warn("Could not resolve the tab");
      return ERROR;
    }

    // has the tab the same originating team?
    if (tab.getTeam() != null && !tab.getTeam().equals(groupId)) {
      logger.warn("{} tried to assign a snapshot to another team than it originated from", owner);
      return ERROR;
    }

    // Get the group
    GroupList groupList = getGroupsForLoggedInPerson(request);
    Group group = groupList.getGroupById(groupId);

    // get the persons
    @SuppressWarnings("unchecked")
    List<Person> persons = (List<Person>) request.getSession().getAttribute(
            LoginInterceptor.INVITEES);
    // TODO remove Hack for performance testing
    persons.add(owner);

    List<Person> inviteesList = addPeople(persons, invitees);

    // Create a new sharedTab and set the attributes
    SharedTab sharedTab = new SharedTab();
    sharedTab.setSharedBy(owner.getId());
    sharedTab.setTimestamp(System.currentTimeMillis());
    sharedTab.setSharedByDisplayName(owner.getDisplayName());
    sharedTab.setTeamTitle(group.getTitle());
    ClonedTab cloneTab = tabService.cloneTab(tab, getGadgets(gadgets));
    sharedTab.setPrototype(cloneTab);

    // Set the invitees
    List<Invite> invites = setInvitees(sharedTab, inviteesList);

    // Save
    sharedResourceService.saveOrUpdate(sharedTab);

    // Create activity
    createActivity(owner, group, tab);

    // Send mail to all invitees
    sendMails(invites, persons, owner, message, locale);

    return SUCCES;
  }

  /*
   * Creates the activity for a shared tab.
   */
  private void createActivity(Person owner, Group group, Tab tab) {
    MessageFormat formatter = new MessageFormat(INVITE_ACTIVITY);
    Object[] messageValues = { tab.getName() };
    Activity activity = new ActivityImpl(null, owner.getId());
    activity.setAppId(APP_ID_INVITE);
    String activityMessage = formatter.format(messageValues);
    activity.setTitle(activityMessage);
    activity.setBody(activityMessage);
    activityService.createActivity(
            new UserId(UserId.Type.userId, owner.getId()),
            new GroupId(GroupId.Type.groupId, group.getId()),
            APP_ID_INVITE, null, activity, null);
  }

  private List<Gadget> getGadgets(String[] gadgets) {
    if (ArrayUtils.isEmpty(gadgets)) {
      return null;
    }
    List<Gadget> result = new ArrayList<Gadget>();
    for (int i = 0; i < gadgets.length; i++) {
      Gadget gadget = gadgetService.findById(Long.parseLong(gadgets[i]));
      result.add(gadget);
    }
    return result;
  }

  private List<Person> removeOwner(List<Person> members, String idToRemove) {
    List<Person> result = new ArrayList<Person>();
    for (Person person : members) {
      if (!person.getId().equals(idToRemove)) {
        result.add(person);
      }
    }
    return result;
  }

  private List<Person> addPeople(List<Person> persons, String[] personsToAdd) {
    List<Person> result = new ArrayList<Person>();
    for (String personToAdd : personsToAdd) {
      result.add(getPersonById(persons, personToAdd));
    }
    return result;
  }

  private List<Invite> setInvitees(SharedResource sharedTab,
                                   List<Person> persons) {
    List<Invite> invites = new ArrayList<Invite>();
    for (Person person : persons) {
      Invite invite = new Invite();
      invite.setInvitee(person.getId());
      invite.setEmail(person.getEmail());
      invite.setStatus(InviteStatus.OPEN);
      invite.setSharedResource(sharedTab);
      sharedTab.addInvite(invite);
      invites.add(invite);
    }

    return invites;
  }

  private void sendMails(List<Invite> invites, List<Person> persons,
                         Person inviter, String textMessage, Locale locale) throws MailException,
          IOException, ProtocolException, InterruptedException, ExecutionException {
    String template = getTemplate(locale);

    if (textMessage == null) {
      textMessage = "";
    }

    MessageFormat formatter = new MessageFormat(template);

    for (Invite invite : invites) {
      Person invitee = getPersonById(persons, invite.getInvitee());
      SimpleMailMessage message = new SimpleMailMessage();
      message.setFrom(INVITE_FROM_ADRESS);
      message.setSubject(INVITE_MESSAGE_SUBJECT);
      message.setTo(invite.getEmail());
      Object[] messageValues = { invitee.getGivenName(),
              inviter.getGivenName(),
              invite.getSharedResource().getName(), textMessage,
              environment.getTargetServerBase() };
      message.setText(formatter.format(messageValues));

      mailService.sendAsync(message);
    }
  }

  private String getTemplate(Locale locale) throws IOException {
    InputStream input;
    if (locale.getLanguage().equals(Locale.ENGLISH)) {
      input = new ClassPathResource(TEMPLATE_INVITES_EMAIL_PLAIN_EN)
              .getInputStream();
    } else {
      input = new ClassPathResource(TEMPLATE_INVITES_EMAIL_PLAIN_NL)
              .getInputStream();
    }
    return input == null ? "" : IOUtils.toString(input);
  }

  private Person getPersonById(List<Person> persons, String personId) {
    for (Person person : persons) {
      if (person.getId().equals(personId)) {
        return person;
      }
    }
    return null;
  }

  private class TabNotFoundException extends RuntimeException {
    public TabNotFoundException(String message) {
      super(message);
    }
  }
}
