/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.domain;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;

import nl.surfnet.coin.shared.domain.DomainObject;

import org.hibernate.annotations.Proxy;
import org.hibernate.annotations.Type;

/**
 * An OpenSocial gadget on a tab.
 */
@SuppressWarnings("serial")
@Entity
@Proxy(lazy = false)
public class Gadget extends DomainObject {
  
	/**
	 * Determines what the order of a gadget is.
	 */
  @Column(name = "gadget_order")
	private int gadgetOrder;
  
	@Column(name = "gadget_column")
	private int gadgetColumn;
	
	@ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "tab_id")
	private Tab tab;

	/**
   * Time of creation
   */
	@Column(name = "creation_timestamp")
  private Long creationTimestamp = System.currentTimeMillis();
	
	/**
	 * The definition on which this gadget is based.
	 */
	@ManyToOne(fetch = FetchType.LAZY)
	private GadgetDefinition definition;

	/**
	 * Contains the user preferences of this gadget.
	 */
	@Column(length = 4000)
	private String prefs;

	/**
	 * Whether the user gave permission to this gadget to access its personal
	 * information.
	 */
	@Type(type = "true_false")
	private boolean hasPermission;
	
	// //
	
	public void setOrder(int order) {
		this.gadgetOrder = order;
	}

	public int getOrder() {
		return gadgetOrder;
	}
	
  public void setColumn(int column) {
    this.gadgetColumn = column;
  }

  public int getColumn() {
    return gadgetColumn;
  }

	public void setDefinition(GadgetDefinition definition) {
		this.definition = definition;
	}

	public GadgetDefinition getDefinition() {
		return definition;
	}

	public void setPrefs(String prefs) {
		this.prefs = prefs;
	}

	public String getPrefs() {
		return prefs;
	}

	public void setHasPermission(boolean hasPermission) {
		this.hasPermission = hasPermission;
	}

	public boolean isHasPermission() {
		return hasPermission;
	}

  /**
   * @return the tab
   */
  public Tab getTab() {
    return tab;
  }

  /**
   * @param tab the tab to set
   */
  public void setTab(Tab tab) {
    this.tab = tab;
  }

  /**
   * Parses the prefs and adds or replaces the groupContext.
   * @param team The team that this gadget should be assigned to.
   */
  public void setTeam(String team) {
    String newPrefs = "";
    
    boolean set = false;
    if(prefs != null) {
      String[] pairs = prefs.split("&");
      for(String pair : pairs) {
        if(!newPrefs.equals("")) {
          newPrefs += "&";
        }
        String[] keyValue = pair.split("=");
        if(keyValue[0].equals("groupContext")) {
          newPrefs += "groupContext=" + (team == null ? "" : team);
          set = true;
        } else {
          newPrefs += pair;
        }
      }
    }
    if(!set) {
      if(!newPrefs.equals("")) {
        newPrefs += "&";
      }
      newPrefs += "groupContext=" + (team == null ? "" : team);
    }
    prefs = newPrefs;
  }

  /**
   * @return the creationTimestamp
   */
  public Long getCreationTimestamp() {
    return creationTimestamp;
  }

  /**
   * @param creationTimestamp the creationTimestamp to set
   */
  public void setCreationTimestamp(Long creationTimestamp) {
    this.creationTimestamp = creationTimestamp;
  }
  
}
