/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.domain;

import java.util.Date;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Table;

import nl.surfnet.coin.shared.domain.DomainObject;

import org.hibernate.annotations.Proxy;
import org.hibernate.annotations.Type;

/**
 * The definition of a gadget. Contains metadata about a gadget.
 */
@SuppressWarnings("serial")
@Entity
@Table(name = "gadgetdefinition")
@Proxy(lazy = false)
public class GadgetDefinition extends DomainObject {

	private static final String TRUE_FALSE = "true_false";

  /**
	 * The URL of the gadget XML file.
	 */
	@Column(unique = true)
	private String url;

	/**
	 * A timestamp of when this gadget was first added to the system.
	 */
	private Date added;

	/**
	 * How many times this gadget was installed.
	 */
	private int installCount;

	/**
	 * Whether this gadget was approved by SURFnet or not.
	 */
	@Type(type = TRUE_FALSE)
	private boolean approved;

	/**
	 * Whether this gadget supports federated Single Sign On.
	 */
	@Type(type = TRUE_FALSE)
	private boolean supportsSSO;

	/**
	 * Whether this gadget can handle groups.
	 */
	@Type(type = TRUE_FALSE)
	private boolean supportsGroups;

  /**
   * Whether the gadget was added as custom gadget (e.g. by a user)
   */
  @Type(type = TRUE_FALSE)
  private boolean customGadget;

	// The following fields should all be read from the metadata of the xml:

  @Column(nullable = false)
	private String title = "Untitled";

	@Column(length = 1024)
	private String description;

	private String author;

	private String authorEmail;

	/**
	 * The URL of the screenshot of the gadget. (should be 280px wide)
	 */
	private String screenshot;

	/**
	 * The URL of a thumbnail of the gadget. (should be 120x60px)
	 */
	private String thumbnail;

	/**
	 * Status of the gadget. (eg. published/unpublished/test)
	 */
	private GadgetDefinitionStatus status = GadgetDefinitionStatus.PUBLISHED;
	
	/**
   * Whether the gadget should appear on the fixed Tab
   */
  @Type(type = TRUE_FALSE)
  private boolean fixedTabGadget;
	
	// //

	public void setUrl(String url) {
		this.url = url;
	}

	public String getUrl() {
		return url;
	}

	public void setAdded(Date added) {
		this.added = added;
	}

	public Date getAdded() {
		return added;
	}

	public void setInstallCount(int installCount) {
		this.installCount = installCount;
	}

	public int getInstallCount() {
		return installCount;
	}

	public void setApproved(boolean approved) {
		this.approved = approved;
	}

	public boolean isApproved() {
		return approved;
	}

	public void setSupportsSSO(boolean supportsSSO) {
		this.supportsSSO = supportsSSO;
	}

	public boolean isSupportsSSO() {
		return supportsSSO;
	}

	public void setSupportsGroups(boolean supportsGroups) {
		this.supportsGroups = supportsGroups;
	}

	public boolean isSupportsGroups() {
		return supportsGroups;
	}

	public void setTitle(String title) {
		this.title = title;
	}

	public String getTitle() {
		return title;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getDescription() {
		return description;
	}

	public void setAuthor(String author) {
		this.author = author;
	}

	public String getAuthor() {
		return author;
	}

	public void setAuthorEmail(String authorEmail) {
		this.authorEmail = authorEmail;
	}

	public String getAuthorEmail() {
		return authorEmail;
	}

	public void setScreenshot(String screenshot) {
		this.screenshot = screenshot;
	}

	public String getScreenshot() {
		return screenshot;
	}

	public void setThumbnail(String thumbnail) {
		this.thumbnail = thumbnail;
	}

	public String getThumbnail() {
		return thumbnail;
	}

  /**
   * @return
   */
  public Gadget protoType() {
    Gadget gadget = new Gadget();
    gadget.setDefinition(this);
    return gadget;
  }

  public void setStatus(GadgetDefinitionStatus status) {
    this.status = status;
  }

  public GadgetDefinitionStatus getStatus() {
    return status;
  }

  public boolean isCustomGadget() {
    return customGadget;
  }

  public void setCustomGadget(boolean customGadget) {
    this.customGadget = customGadget;
  }

  /**
   * @return the fixedTabGadget
   */
  public boolean isFixedTabGadget() {
    return fixedTabGadget;
  }

  /**
   * @param fixedTabGadget the fixedTabGadget to set
   */
  public void setFixedTabGadget(boolean fixedTabGadget) {
    this.fixedTabGadget = fixedTabGadget;
  }
}
