/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.domain;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.DiscriminatorType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Transient;
import javax.validation.constraints.NotNull;

import org.hibernate.annotations.Proxy;
import org.hibernate.annotations.Type;

/**
 * Represents a tab which can contain gadgets.
 */
@SuppressWarnings("serial")
@Entity
@Inheritance(strategy = InheritanceType.SINGLE_TABLE)
@DiscriminatorColumn(name = "OBJECT_TYPE", discriminatorType = DiscriminatorType.STRING)
@Proxy(lazy = false)
public class Tab extends AbstractTab {
  
  /**
   * The list of gadgets on this tab.
   */
  @OneToMany(cascade = CascadeType.ALL, fetch = FetchType.EAGER, mappedBy = "tab")
  @OrderBy("gadgetOrder")
  private List<Gadget> gadgets;

  /**
   * The owner (Person) of this tab.
   */
  @NotNull
  @Column(name = "owner_id")
  private String owner;

  /**
   * When this is false it will only show up in the All Tabs overview screen.
   */
  @Type(type = "true_false")
  private Boolean favorite = false;

  /**
   * Denotes the order of the favorite tabs.
   */
  @Column(name = "tab_order")
  private int order;

  /**
   * The team to which this tab is assigned or null if it's not assigned.
   */
  private String team;

  /**
   * Redundant team title for performance optimalisation
   */
  private String teamTitle;

  /**
   * @return the team
   */
  public String getTeam() {
    return team;
  }

  /**
   * @param team
   *          the team to set
   */
  public void setTeam(String team) {
    this.team = team;
    if(gadgets != null) {
      for(Gadget g : gadgets) {
        g.setTeam(team);
      }
    }
  }

  public boolean addGadget(Gadget gadget) {
    if (gadgets == null) {
      gadgets = new ArrayList<Gadget>();
    }
    gadget.setTab(this);
    return gadgets.add(gadget);
  }

  /**
   * @return the gadgets
   */
  public List<Gadget> getGadgets() {
    return gadgets;
  }

  /**
   * @param gadgets
   *          the gadgets to set
   */
  private void setGadgets(List<Gadget> gadgets) {
    this.gadgets = gadgets;
  }

  /**
   * @return the owner
   */
  public String getOwner() {
    return owner;
  }

  /**
   * @param owner
   *          the owner to set
   */
  public void setOwner(String owner) {
    this.owner = owner;
  }

  /**
   * @return the favorite
   */
  public Boolean getFavorite() {
    return favorite;
  }

  /**
   * @param favorite
   *          the favorite to set
   */
  public void setFavorite(Boolean favorite) {
    this.favorite = favorite;
  }

  /**
   * @return the order
   */
  public int getOrder() {
    return order;
  }

  /**
   * @param order
   *          the order to set
   */
  public void setOrder(int order) {
    this.order = order;
  }
  
  /**
   * 
   * @return teamTitle
   *          the team title
   */
  public String getTeamTitle() {
    return teamTitle;
  }

  /**
   * @param title
   */
  public void setTeamTitle(String title) {
    this.teamTitle = title;

  }

}
