/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.service.impl;

import nl.surfnet.coin.portal.domain.GadgetDefinition;
import nl.surfnet.coin.portal.domain.GadgetDefinitionStatus;
import nl.surfnet.coin.portal.service.GadgetDefinitionService;
import nl.surfnet.coin.shared.service.GenericServiceHibernateImpl;
import nl.surfnet.coin.shared.service.SortOrder;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@Component("gadgetDefinitionService")
public class GadgetDefinitionServiceHibernateImpl extends
    GenericServiceHibernateImpl<GadgetDefinition> implements
    GadgetDefinitionService {

  public GadgetDefinitionServiceHibernateImpl() {
    super(GadgetDefinition.class);
  }

  @Override
  public List<GadgetDefinition> searchByWildCard(String query) {
    Criterion queryCrit = Restrictions.ilike("title", query);
    Criterion statusCrit = Restrictions.eq("status", GadgetDefinitionStatus.PUBLISHED);
    return findByCriteria(queryCrit, statusCrit);
  }

  @Override
  public List<GadgetDefinition> searchByWildCardExcludeCustom(String query) {
    Criterion queryCrit = Restrictions.ilike("title", query);
    Criterion statusCrit = Restrictions.eq("status", GadgetDefinitionStatus.PUBLISHED);
    Criterion customCrit = Restrictions.ne("customGadget", Boolean.TRUE);
    return findByCriteria(queryCrit, statusCrit, customCrit);
  }

  @Override
  public List<GadgetDefinition> findAllPublished() {
    Criterion statusCrit = Restrictions.eq("status", GadgetDefinitionStatus.PUBLISHED);
    return findByCriteria(statusCrit);
  }

  @Override
  public List<GadgetDefinition> findAllPublishedExcludeCustom() {
    Criterion statusCrit = Restrictions.eq("status", GadgetDefinitionStatus.PUBLISHED);
    Criterion customCrit = Restrictions.ne("customGadget", Boolean.TRUE);
    return findByCriteria(statusCrit, customCrit);
  }

  @Override
  public List<GadgetDefinition> findAllPublishedExcludeCustomOrderByPopularity(SortOrder sortOrder) {
    Criterion statusCrit = Restrictions.eq("status", GadgetDefinitionStatus.PUBLISHED);
    Criterion customCrit = Restrictions.ne("customGadget", Boolean.TRUE);
    List<Criterion> criterionList = new ArrayList<Criterion>();
    Collections.addAll(criterionList, statusCrit, customCrit);

    List<Order> orderByList = new ArrayList<Order>();
    if (sortOrder.equals(SortOrder.ASCENDING)) {
      orderByList.add(Order.asc("installCount"));
    } else if (sortOrder.equals(SortOrder.DESCENDING)) {
      orderByList.add(Order.desc("installCount"));
    }
    return findByCriteriaOrdered(criterionList, orderByList);
  }

  @Override
  public GadgetDefinition searchByUrl(String url) {
    Criterion urlCrit = Restrictions.eq("url", url);
    List<GadgetDefinition> gadgetDefinitions = findByCriteria(urlCrit);
    return gadgetDefinitions.isEmpty() ? null : gadgetDefinitions.get(0);
  }

  /* (non-Javadoc)
   * @see nl.surfnet.coin.portal.service.GadgetDefinitionService#findAllFixedTabGadgets()
   */
  @Override
  public List<GadgetDefinition> findAllFixedTabGadgets() {
    Criterion customCrit = Restrictions.eq("fixedTabGadget", Boolean.TRUE);
    return findByCriteria(customCrit);
  }

}
