/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.service.impl;

import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.service.GadgetService;
import nl.surfnet.coin.shared.service.GenericServiceHibernateImpl;
import org.hibernate.Criteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.util.List;

@Component("gadgetService")
public class GadgetServiceHibernateImpl extends
    GenericServiceHibernateImpl<Gadget> implements GadgetService {

  public GadgetServiceHibernateImpl() {
    super(Gadget.class);
  }

  /*
   * (non-Javadoc)
   * 
   * @see nl.surfnet.coin.portal.service.GadgetService#getCountByTab(nl.surfnet
   * .coin.portal.domain.Tab)
   */
  @Override
  public int getCountByTab(Tab tab) {
    return ((Integer) createCriteria().setProjection(Projections.rowCount())
        .add(Restrictions.eq("tab", tab))
        .setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY).list().get(0))
        .intValue();

  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.GadgetService#getCountByTabColumn(nl.surfnet
   * .coin.portal.domain.Tab, int)
   */
  @Override
  public int getCountByTabColumn(Tab tab, int column) {
    return ((Integer) createCriteria().setProjection(Projections.rowCount())
        .add(Restrictions.eq("tab", tab))
        .add(Restrictions.eq("gadgetColumn", column))
        .setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY).list().get(0))
        .intValue();
  }

  /*
   * (non-Javadoc)
   * 
   * @see nl.surfnet.coin.portal.service.GadgetService#getByColumn(nl.surfnet
   * .coin.portal.domain.Tab, int)
   */
  @SuppressWarnings("unchecked")
  @Override
  public List getByTabColumn(Tab tab, int column) {
    return createCriteria().addOrder(Order.asc("gadgetOrder"))
        .add(Restrictions.eq("tab", tab))
        .add(Restrictions.eq("gadgetColumn", column))
        .setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY).list();
  }

  /*
   * (non-Javadoc)
   * 
   * @see nl.surfnet.coin.portal.service.GadgetService#reorderGadgets(java.lang
   * .String[], java.lang.String[], nl.surfnet.coin.portal.domain.Tab)
   */
  @Override
  public boolean reorderGadgets(String[] column1, String[] column2, Tab tab) {
    if (ObjectUtils.isEmpty(column1)) {
      column1 = new String[0];
    }

    if (ObjectUtils.isEmpty(column2)) {
      column2 = new String[0];
    }

    // Always first reorder column 1, then column 2
    if (reorderGadgetsForColumn(1, column1, 0)
        && reorderGadgetsForColumn(2, column2, column1.length)) {
      // tabservice.saveOrUpdate(tab);
      return true;
    }

    return false;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.GadgetService#clearFirstSpot(nl.surfnet.
   * coin.portal.domain.Tab)
   */
  @Override
  public void clearFirstSpot(Tab tab) {
    List<Gadget> gadgets = tab.getGadgets();
    for (Gadget gadget : gadgets) {
      gadget.setOrder(gadget.getOrder() + 1);
    }

    saveOrUpdate(gadgets);
  }

  private boolean reorderGadgetsForColumn(int column, String[] newOrder,
      int startOrder) {
    if (newOrder.length > 0) {
      for (int i = 0; i < newOrder.length; i++) {
        Gadget gadget = findById(Long.parseLong(newOrder[i]));
        gadget.setColumn(column);
        gadget.setOrder(i + startOrder);
      }
    }

    return true;
  }

}
