/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package nl.surfnet.coin.portal.service.impl;

import nl.surfnet.coin.portal.domain.IdentityProvider;
import nl.surfnet.coin.portal.service.IdentityProviderService;
import nl.surfnet.coin.portal.util.CoinEnvironment;
import nl.surfnet.coin.portal.util.XPathUtil;
import nl.surfnet.coin.shared.domain.ErrorMail;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.io.InputStream;
import java.net.InetAddress;
import java.net.URL;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * {@link IdentityProviderService}
 */
@Component("identityProviderService")
public class IdentityProviderServiceImpl implements IdentityProviderService {

  private Logger logger = Logger.getLogger(this.getClass());

  @Autowired
  private CoinEnvironment environment;

  @Autowired
  private ErrorMessageMailer errorMessageMailer;

  private static final XPathUtil XPATHUTIL = new XPathUtil();

  public void setEnvironment(CoinEnvironment environment) {
    this.environment = environment;
  }

  /**
   * {@inheritDoc}
   * <p/>
   * Gets the metadata from the configured location and returns a List of
   * {@link IdentityProvider}s
   *
   * @return List of {@link IdentityProvider}s, can be empty
   */
  @Override
  public List<IdentityProvider> getIdps() {
    List<IdentityProvider> identityProviders = new ArrayList<IdentityProvider>();

    InputStream in;
    try {
      in = new URL(environment.getIdpMetadataUrl()).openStream();
    } catch (Exception e) {
      String shortMessage = "Cannot fetch Idp Metadata from "
              + environment.getIdpMetadataUrl();
      sendErrorMail(shortMessage, e);
      return identityProviders;
    }

    // Get the nodelist containing the entityID's
    NodeList nodeList;
    try {
      // warning: XpathUtil#getNodes() closes the inputStream
      nodeList = XPATHUTIL.getNodes(in, idpXMLNamespaces(),
              "/md:EntitiesDescriptor/md:EntityDescriptor");
    } catch (Exception e) {
      String shortMessage = "Cannot fetch Idp Metadata from "
              + environment.getIdpMetadataUrl();
      sendErrorMail(shortMessage, e);
      return identityProviders;
    }

    // Iterate over the found entity descriptors
    addIdentityProviders(identityProviders, nodeList);

    return identityProviders;
  }

  /**
   * Adds {@link IdentityProvider} to the list
   *
   * @param identityProviders List of IdentityProvider's
   * @param nodeList          {@link NodeList} of which each Node represents an IdentityProvider
   */
  private void addIdentityProviders(List<IdentityProvider> identityProviders,
                                    NodeList nodeList) {
    String entityId = null;
    for (int position = 0; position < nodeList.getLength(); position++) {
      int xPathPos = position + 1;
      IdentityProvider idp = new IdentityProvider();
      try {
        // Encode the entityId's, otherwise Shibboleth will break.
        Node idpNode = nodeList.item(position);

        // Skip if entitydescriptor is not an IdP
        if (XPATHUTIL.getNode(idpNode, idpXMLNamespaces(),
                "/md:EntitiesDescriptor/md:EntityDescriptor[" + xPathPos
                        + "]//md:IDPSSODescriptor") != null) {

          entityId = idpNode.getAttributes().getNamedItem("entityID")
                  .getNodeValue();
          idp.setEntityId(URLEncoder.encode(entityId, "UTF-8"));

          // Get the display names
          NodeList displayNames = XPATHUTIL.getNodes(idpNode,
                  idpXMLNamespaces(), "/md:EntitiesDescriptor/md:EntityDescriptor["
                  + xPathPos + "]//mdui:DisplayName");
          if (displayNames == null || displayNames.getLength() == 0) {
            throw new RuntimeException("No display name Specified for IdP: "
                    + entityId);
          }
          addDisplayNames(idp, displayNames);

          // Get the logo
          Node logo = XPATHUTIL.getNode(idpNode, idpXMLNamespaces(),
                  "/md:EntitiesDescriptor/md:EntityDescriptor[" + xPathPos
                          + "]//mdui:Logo");
          if (logo != null && logo.hasChildNodes()) {
            idp.setLogo(logo.getFirstChild().getNodeValue());
          }

          // Get the GEOlocation
          Node geoLocation = XPATHUTIL.getNode(idpNode, idpXMLNamespaces(),
                  "/md:EntitiesDescriptor/md:EntityDescriptor[" + xPathPos
                          + "]//mdui:GeolocationHint");
          if (geoLocation != null && geoLocation.hasChildNodes()) {
            idp.setGeoLocationHint(geoLocation.getFirstChild().getNodeValue());
          }

          // Get the Keywords
          NodeList keywords = XPATHUTIL.getNodes(idpNode, idpXMLNamespaces(),
                  "/md:EntitiesDescriptor/md:EntityDescriptor[" + xPathPos
                          + "]//mdui:Keywords");
          addKeywords(idp, keywords);

          identityProviders.add(idp);
        }
      } catch (Exception e) {
        String shortMessage = MessageFormat.format("Could not parse Idp "
                + entityId + " at position {0} from url {1}", xPathPos,
                environment.getIdpMetadataUrl());
        sendErrorMail(shortMessage, e);
      }
    }
  }

  /**
   * Handles the displaynames for each language of an IdentityProvider
   *
   * @param idp          {@link IdentityProvider}
   * @param displayNames {@link NodeList} that contains the translations of the displayname
   */
  private void addDisplayNames(IdentityProvider idp, NodeList displayNames) {
    for (int j = 0; j < displayNames.getLength(); j++) {
      Node item = displayNames.item(j);
      String language = item.getAttributes().getNamedItem("xml:lang")
              .getNodeValue();
      Node firstChild = item.getFirstChild();
      if (firstChild != null) {
        String displayName = firstChild.getNodeValue();
        idp.setDisplayName(language, displayName);
      }

    }
  }

  private void addKeywords(IdentityProvider idp, NodeList keywordsList) {
    for (int j = 0; j < keywordsList.getLength(); j++) {
      Node item = keywordsList.item(j);
      String language = item.getAttributes().getNamedItem("xml:lang")
              .getNodeValue();
      Node firstChild = item.getFirstChild();
      if (firstChild != null) {
        String[] keywords = firstChild.getNodeValue().split(" ");
        idp.setKeyword(language, keywords);
      }
    }
  }

  private static Map<String, String> idpXMLNamespaces() {
    Map<String, String> namespaces = new HashMap<String, String>();
    namespaces.put("md", "urn:oasis:names:tc:SAML:2.0:metadata");
    namespaces.put("mdui", "urn:oasis:names:tc:SAML:metadata:ui");
    return namespaces;
  }

  private void sendErrorMail(String shortMessage, Exception e) {
    String errorMessage = "An exception occured in the Conext portal IdentityProviderService while<br/>" +
            "trying to get the metadata from the Identity provider.<br/><br/>" +
            getClass().getSimpleName() + "<br/>" + shortMessage;
    ErrorMail errorMailDetails = new ErrorMail(shortMessage, errorMessage, e.getMessage(), getHost(), "Portal");
    errorMessageMailer.sendErrorMail(errorMailDetails);
    logger.error(shortMessage, e);
  }

  private String getHost() {
    try {
      return InetAddress.getLocalHost().toString();
    } catch (UnknownHostException e) {
      return "UNKNOWN";
    }
  }

  public void setErrorMessageMailer(ErrorMessageMailer errorMessageMailer) {
    this.errorMessageMailer = errorMessageMailer;
  }
}
