/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package nl.surfnet.coin.portal.service.impl;

import nl.surfnet.coin.portal.domain.Invite;
import nl.surfnet.coin.portal.domain.InviteStatus;
import nl.surfnet.coin.portal.service.InviteService;
import nl.surfnet.coin.shared.service.GenericServiceHibernateImpl;
import org.hibernate.Criteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * Common shared code for {@link nl.surfnet.coin.portal.domain.SharedGadget}
 * and {@link nl.surfnet.coin.portal.domain.SharedTab}.
 */
@Component("inviteService")
public class InviteServiceHibernateImpl extends
        GenericServiceHibernateImpl<Invite> implements InviteService {

  /**
   * The amount of miliseconds that count as recent (30 days).
   */
  public final static long RECENT_MS = 1000L * 60 * 60 * 24 * 30;

  /**
   */
  public InviteServiceHibernateImpl() {
    super(Invite.class);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.SharedGadgetService#findByInvitee(org.apache
   * .shindig.social.opensocial.model.Person)
   */
  @SuppressWarnings("unchecked")
  public List<Invite> findRecentByInvitee(String inviteeId, InviteStatus status) {
    Criteria criteria = createCriteria().add(
            Restrictions.eq("invitee", inviteeId));

    if (status != null) {
      criteria.add(Restrictions.eq("status", status));
    }

    long recent = System.currentTimeMillis() - RECENT_MS;
    criteria.add(Restrictions.gt("creationTimestamp", recent));
    criteria.addOrder(Order.desc("creationTimestamp"));

    criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
    return criteria.list();
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.SharedTabService#getCountByInvitee(org.apache
   * .shindig.social.opensocial.model.Person)
   */
  public int getCountRecentByInvitee(String ownerId, InviteStatus status) {
    Criteria criteria = createCriteria().setProjection(Projections.rowCount())
            .add(Restrictions.eq("invitee", ownerId));

    if (status != null) {
      criteria.add(Restrictions.eq("status", status));
    }

    long recent = System.currentTimeMillis() - RECENT_MS;
    criteria.add(Restrictions.gt("creationTimestamp", recent));

    return ((Integer) criteria.setResultTransformer(
            Criteria.DISTINCT_ROOT_ENTITY).list().get(0)).intValue();
  }

}
