/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.service.impl;

import nl.surfnet.coin.portal.domain.ClonedTab;
import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.service.TabService;
import nl.surfnet.coin.portal.util.TabOrderComparator;
import nl.surfnet.coin.shared.service.GenericServiceHibernateImpl;
import org.apache.shindig.social.opensocial.model.Person;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;

@Component("tabService")
public class TabServiceHibernateImpl extends GenericServiceHibernateImpl<Tab>
        implements TabService {

  private final TabOrderComparator tabOrderComparator = new TabOrderComparator();

  /**
   */
  public TabServiceHibernateImpl() {
    super(Tab.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Tab> findFavorites(String ownerId, boolean favorites) {
    SimpleExpression favoriteEq = Restrictions.eq("favorite", Boolean
            .valueOf(favorites));
    SimpleExpression ownerEq = Restrictions.eq("owner", ownerId);
    return findByCriteria(favoriteEq, ownerEq);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.TabService#reorderFavoriteTabs(java.lang
   * .String[], org.apache.shindig.social.opensocial.model.Person)
   */
  @Override
  public boolean reorderTabs(String[] newOrder, String ownerId, boolean favorites) {
    List<Tab> tabs = findByCriteria(Restrictions.in("id", convertStringToId(newOrder)), Restrictions.eq("owner", ownerId));
    for (int i = 0; i < newOrder.length; i++) {
      Tab tab = tabs.get(i);
      tab.setOrder(retrieveOrder(tab, newOrder));
      tab.setFavorite(favorites);
    }
    saveOrUpdate(tabs);
    return true;
  }

  private int retrieveOrder(Tab tab, String[] newOrder) {
    for (int i = 0; i < newOrder.length; i++) {
      if (tab.getId().equals(Long.parseLong(newOrder[i]))) {
        return i;
      }
    }
    return 0;
  }

  private Long[] convertStringToId(String[] newOrder) {
    Long[] result = new Long[newOrder.length];
    ;
    for (int i = 0; i < newOrder.length; i++) {
      result[i] = Long.parseLong(newOrder[i]);
    }
    return result;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.TabService#getCountByOwner(org.apache.shindig
   * .social.opensocial.model.Person, boolean)
   */
  @Override
  public int getCountByOwner(String ownerId, boolean favorites) {
    return DataAccessUtils.intResult(getSession().createQuery(
            "select count(*) from " + getPersistentClass().getName()
                    + " where owner = :owner and favorite = :favorite").setString(
            "owner", ownerId).setBoolean("favorite", favorites).list());
  }

  /*
  * (non-Javadoc)
  *
  * @see
  * nl.surfnet.coin.portal.service.TabService#getCountByOwner(org.apache.shindig
  * .social.opensocial.model.Person, boolean)
  */
  @Override
  public ClonedTab cloneTab(Tab tab, List<Gadget> sharedGadgets) {
    ClonedTab clonedTab = new ClonedTab();
    clonedTab.setName(tab.getName());
    clonedTab.setTeam(tab.getTeam());
    if (!CollectionUtils.isEmpty(sharedGadgets)) {
      for (Gadget gadget : tab.getGadgets()) {
        // Only add gadgets that are on the tab.
        if (sharedGadgets.contains(gadget)) {
          Gadget clonedGadget = new Gadget();
          clonedGadget.setColumn(gadget.getColumn());
          clonedGadget.setDefinition(gadget.getDefinition());
          clonedGadget.setHasPermission(false);
          clonedGadget.setOrder(gadget.getOrder());
          clonedGadget.setPrefs(gadget.getPrefs());
          clonedGadget.setTab(clonedTab);
          clonedTab.addGadget(clonedGadget);
        }
      }
    }

    saveOrUpdate(clonedTab);
    return clonedTab;
  }

  /* (non-Javadoc)
   * @see nl.surfnet.coin.portal.service.impl.GenericServiceHibernateImpl#findByCriteria(org.hibernate.criterion.Criterion[])
   */
  @Override
  protected List<Tab> findByCriteria(Criterion... criterion) {
    List<Tab> tabs = super.findByCriteria(criterion);
    //sort on tabOrder
    Collections.sort(tabs, tabOrderComparator);
    return tabs;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.TabService#getTabsByPerson(org.apache.shindig
   * .social.opensocial.model.Person)
   */
  @Override
  public List<Tab> getTabsByOwner(String ownerId) {
    SimpleExpression ownerEq = Restrictions.eq("owner", ownerId);
    return findByCriteria(ownerEq);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Tab findByIdAndOwner(Long id, String ownerId) {
    if (id == null || ownerId == null) {
      return null;
    }
    Criterion idCrit = Restrictions.eq("id", id);
    Criterion ownerCrit = Restrictions.eq("owner", ownerId);
    final List<Tab> results = findByCriteria(idCrit, ownerCrit);
    return CollectionUtils.isEmpty(results) ? null : results.get(0);
  }

}
