/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.portal.util;


import org.apache.shindig.common.util.ImmediateFuture;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.social.core.model.GroupImpl;
import org.apache.shindig.social.core.model.NameImpl;
import org.apache.shindig.social.core.model.PersonImpl;
import org.apache.shindig.social.opensocial.model.Group;
import org.apache.shindig.social.opensocial.model.Name;
import org.apache.shindig.social.opensocial.model.Person;
import org.apache.shindig.social.opensocial.spi.GroupId;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Future;

/**
 * Historically the Portal uses the Shindig Person and Group models. This class provides convertors to convert the
 * standard Opensocial model {@link org.opensocial.models.Group} & {@link org.opensocial.models.Person} to the Shindig
 * Model {@link org.apache.shindig.social.opensocial.model.Person}
 * & {@link org.apache.shindig.social.opensocial.model.Group}
 */
public class OpenSocialModelConverter {

  public static Person convertOpenSocialPersonToShindigPerson(org.opensocial.models.Person openSocialPerson) {
    Name name = new NameImpl(openSocialPerson.getDisplayName());
    Person shindigPerson = new PersonImpl(openSocialPerson.getId(), openSocialPerson.getDisplayName(), name);
    shindigPerson.setTags(getPersonTags(openSocialPerson));
//    shindigPerson.setVoot_membership_role();
    return shindigPerson;
  }

  public static Group convertOpenSocialGroupToShindigGroup(org.opensocial.models.Group openSocialGroup) {
    Group shindigGroup = new GroupImpl();
    shindigGroup.setId(new GroupId(GroupId.Type.groupId, openSocialGroup.getId()));
    shindigGroup.setDescription(openSocialGroup.getDescription());
    shindigGroup.setTitle(openSocialGroup.getTitle());
    return shindigGroup;
  }

  private static List<String> getPersonTags(org.opensocial.models.Person openSocialPerson) {
    List<String> tags = new ArrayList<String>();
    tags.add(openSocialPerson.isGuest() ? "guest" : "member");
    return tags;
  }

  public static List<Person> convertOpenSocialPersonListToShindigPersonList(List<org.opensocial.models.Person> openSocialPersons) {
    List<Person> shindigPersons = new ArrayList<Person>();
    for (org.opensocial.models.Person openSocialPerson : openSocialPersons) {
      shindigPersons.add(convertOpenSocialPersonToShindigPerson(openSocialPerson));
    }
    return shindigPersons;
  }

  public static List<Group> convertOpenSocialGroupListToShindigGroupList(List<org.opensocial.models.Group> openSocialGroups) {
    List<Group> shindigGroups = new ArrayList<Group>();
    for (org.opensocial.models.Group openSocialGroup : openSocialGroups) {
      shindigGroups.add(convertOpenSocialGroupToShindigGroup(openSocialGroup));
    }
    return shindigGroups;
  }

}
