/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class XPathUtil {

  private static DocumentBuilderFactory dbFactory = null;
  private static XPathFactory xpFactory = null;
  private static XPath xpath = null;

  /**
   * Constructor
   */
  public XPathUtil() {
    dbFactory = DocumentBuilderFactory.newInstance();
    xpFactory = XPathFactory.newInstance();
    xpath = xpFactory.newXPath();
  }

  /**
   * Parses an {@link InputStream} and returns a {@link NodeList} of the given xpath
   *
   * @param in         the {@link InputStream} of the XML structure
   * @param namespaces {@link Map} of namespaces
   * @param xPathStr   String with the xpath expression
   * @return {@link NodeList} that is the result of the xpath expression
   * @throws IOException if the input cannot be read
   * @throws SAXException if the input is malformed XML
   * @throws ParserConfigurationException
   * @throws XPathExpressionException in case of an invalid xpath
   */
  public NodeList getNodes(InputStream in, Map<String, String> namespaces, String xPathStr)
      throws IOException, SAXException, ParserConfigurationException, XPathExpressionException {
    // Parse XML to DOM
    Document doc = getDocument(in);
    return getNodes(doc, namespaces, xPathStr);
  }

  /**
   * Returns a {@link NodeList} for the given xpath under a {@link Node}
   *
   * @param startNode  the {@link Node} that is the root of the xpath
   * @param namespaces {@link Map} of namespaces
   * @param xPathStr   String with the xpath expression
   * @return {@link NodeList} that is the result of the xpath expression
   * @throws XPathExpressionException in case of an invalid xpath
   */
  public NodeList getNodes(Node startNode, Map<String, String> namespaces, String xPathStr)
      throws XPathExpressionException {
    // namespace?
    if (!namespaces.isEmpty()) {
      xpath.setNamespaceContext(new NamespaceResolver().addNamespaces(namespaces));
    }
     XPathExpression expr = xpath.compile(xPathStr);
     return (NodeList) expr.evaluate(startNode, XPathConstants.NODESET);
  }

  /**
   * Returns a single {@link Node} for the given xpath under a {@link Node}
   *
   * @param startNode  the {@link Node} that is the root of the xpath
   * @param namespaces {@link Map} of namespaces
   * @param xPathStr   String with the xpath expression
   * @return {@link Node} that is the result of the xpath expression
   * @throws XPathExpressionException in case of an invalid xpath
   */
  public Node getNode(Node startNode, Map<String, String> namespaces, String xPathStr)
      throws XPathExpressionException {
    // namespace?
    if (!namespaces.isEmpty()) {
      xpath.setNamespaceContext(new NamespaceResolver().addNamespaces(namespaces));
    }
    XPathExpression expr = xpath.compile(xPathStr);
    return (Node) expr.evaluate(startNode, XPathConstants.NODE);
  }


  /**
   * Makes a {@link org.w3c.dom.Document} from an {@link InputStream} and closes the stream
   * @param in {@link InputStream} to parse
   * @return {@link org.w3c.dom.Document}
   * @throws ParserConfigurationException
   * @throws IOException
   * @throws SAXException
   */
  private Document getDocument(InputStream in) throws ParserConfigurationException, IOException, SAXException {
    dbFactory.setNamespaceAware(true);
    dbFactory.setValidating(false);
    DocumentBuilder builder;
    try {
      builder = dbFactory.newDocumentBuilder();
      builder.setEntityResolver(emptyEntityResolver());
      return builder.parse(in);
    } finally {
      if (in != null){
      in.close();
      }
    }
  }

  private static EntityResolver emptyEntityResolver() {
    return new EntityResolver() {
      @Override
      public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
        return new InputSource(new StringReader(""));
      }
    };
  }
}
