/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package nl.surfnet.coin.portal.control;

import nl.surfnet.coin.opensocial.service.GroupService;
import nl.surfnet.coin.opensocial.service.PersonService;
import nl.surfnet.coin.portal.domain.ExternalAwareGroup;
import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.domain.UserPreferences;
import nl.surfnet.coin.portal.interceptor.LoginInterceptor;
import nl.surfnet.coin.portal.service.GadgetService;
import nl.surfnet.coin.portal.service.TabService;
import nl.surfnet.coin.portal.service.UserPreferencesService;
import nl.surfnet.coin.portal.util.GroupList;
import org.opensocial.models.Group;
import org.opensocial.models.Person;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.ui.ModelMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;

/**
 * Base class for {@link org.springframework.stereotype.Controller} that needs
 * access to the {@link GroupService}
 *
 * @author oharsta
 */
public class BaseController {

  @Autowired
  protected TabService tabService;

  @Autowired
  protected GadgetService gadgetService;

  @Autowired
  @Qualifier(value = "opensocialPersonService")
  protected PersonService personService;

  @Autowired
  protected UserPreferencesService userPreferencesService;

  @Autowired
  @Qualifier(value = "opensocialGroupService")
  private GroupService groupService;
  public static Logger logger = LoggerFactory.getLogger(BaseController.class);

  protected GroupList getGroupsForLoggedInPerson(HttpServletRequest request)
          throws InterruptedException, ExecutionException {

    Person person = getPerson(request);
    List<Group> groups = convertGroups(groupService.getGroups(person.getId()));
    return new GroupList(groups);
  }

  /*
   * Convert to ExternalAwareGroup for nicer description
   */
  private List<Group> convertGroups(List<Group> entry) {
    List<Group> result = new ArrayList<Group>();
    for (Group group : entry) {
      result.add(new ExternalAwareGroup(group));
    }
    return result;
  }

  protected boolean isMemberOfGroup(HttpServletRequest request, String teamName) {
    try {
      final GroupList groups = getGroupsForLoggedInPerson(request);
      return groups.getGroupById(teamName) != null;
    } catch (InterruptedException iExp) {
      logger.warn("Error in getting groups for user " + request.getSession().getId(), iExp);
    } catch (ExecutionException eExp) {
      logger.warn("Error in getting groups for user " + request.getSession().getId(), eExp);
    }
    return false;
  }

  protected Gadget getGadget(HttpServletRequest request, final String gadgetId) {
    Long id = Long.parseLong(gadgetId);
    Person owner = getPerson(request);
    return getGadget(owner, id);
  }

  private Gadget getGadget(final Person owner, final Long id) {
    Gadget gadget = gadgetService.findById(id);
    if (gadget == null) {
      logger.info("Could not find gadget with id {}", id);
      return null;
    }
    Tab tab = gadget.getTab();
    if (tab == null) {
      logger.info("Could not lookup tab for gadgetId {}", id);
      return null;
    }
    if (!tab.getOwner().equals(owner.getId())) {
      logger.warn("Person {} doesn't have access to tab {}", owner.getId(), tab.getId());
      return null;
    }
    return gadget;
  }

  protected Tab getTab(HttpServletRequest request, String tabId) {
    Long id = Long.parseLong(tabId);
    Person owner = getPerson(request);
    return tabService.findByIdAndOwner(id, owner.getId());
  }

  protected Person getPerson(HttpServletRequest request) {
    HttpSession session = request.getSession();
    Person person = (Person) session
            .getAttribute(LoginInterceptor.PERSON_SESSION_KEY);
    return person;
  }

  /**
   * Fetch the user's preferences
   *
   * @param request
   * @return
   */
  protected UserPreferences getPreferences(HttpServletRequest request) {
    Person person = getPerson(request);
    if (person == null) {
      logger.info("Couldn't fetch the person");
      return null;
    }
    return userPreferencesService.getPreferences(person.getId());
  }

  /**
   * Populate the model map with the user's preferences.
   *
   * @param modelMap
   * @param request
   */
  protected void populateModelMapWithPreferences(final ModelMap modelMap, final HttpServletRequest request) {
    UserPreferences userPreferences = getPreferences(request);
    if (userPreferences == null) {
      return;
    }
    modelMap.addAttribute("infoSharingGranted",
            userPreferences.isInformationSharingPermissionGranted());
  }
}
