/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package nl.surfnet.coin.portal.control;

import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.GadgetDefinition;
import nl.surfnet.coin.portal.domain.InviteStatus;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.interceptor.LoginInterceptor;
import nl.surfnet.coin.portal.service.GadgetDefinitionService;
import nl.surfnet.coin.portal.service.InviteService;
import nl.surfnet.coin.portal.service.MetadataProvider;
import nl.surfnet.coin.portal.util.CoinEnvironment;
import nl.surfnet.coin.portal.util.GroupList;
import org.apache.shindig.auth.BlobCrypterSecurityToken;
import org.apache.shindig.common.crypto.BasicBlobCrypter;
import org.apache.shindig.common.crypto.BlobCrypterException;
import org.apache.shindig.common.util.Utf8UrlCoder;
import org.opensocial.models.Group;
import org.opensocial.models.Person;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.servlet.http.HttpServletRequest;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * {@link Controller} that handles the home page of a logged in user.
 */
@Controller
public class HomeController extends BaseController {

  @Autowired
  private MetadataProvider metadataProvider;

  @Autowired
  private CoinEnvironment environment;

  @Autowired
  private InviteService inviteService;

  @Autowired
  private GadgetDefinitionService gadgetDefinitionService;

  @RequestMapping("/home.shtml")
  public String start(ModelMap modelMap, HttpServletRequest request)
          throws Exception {
    populateModelMapWithPreferences(modelMap, request);

    // Retrieve the owner of a tab
    Person owner = getPerson(request);

    List<Tab> favoriteTabList = getFavoriteTabList(owner);
    modelMap.addAttribute("favoriteTabList", favoriteTabList);

    // Retrieve the requested tab and add it to the modelMap
    Tab requestedTab = getRequestedTab(request, owner, favoriteTabList);
    if (requestedTab == null) {
      return "redirect:/taboverview.shtml";
    }
    modelMap.addAttribute("requestedTab", requestedTab);

    // Retrieve the requested tab Gadgets and add them to the modelmap
    addGadgetsToModelMap(modelMap, requestedTab, owner);
    addTeamToModelMap(modelMap, requestedTab, getGroupsForLoggedInPerson(request));
    addInvitesToModelMap(modelMap, owner);

    modelMap.addAttribute(LoginInterceptor.PERSON_SESSION_KEY, owner);
    modelMap.addAttribute("environment", environment);

    String display = request.getParameter("display");
    if (display != null) {
      modelMap.addAttribute("display", display);
    }

    return "home";
  }

  private List<Tab> getFavoriteTabList(Person owner) {
    List<Tab> favoriteTabList = tabService.findFavorites(owner.getId(), true);

    if (!containsFixedTab(favoriteTabList)) {
      Tab tab = createFixedTab(owner);
      if (tab != null) {
        favoriteTabList.add(tab);
      }
    }
    return favoriteTabList;
  }

  /*
   * Create a Fixed Tab for this person (who apparently logs in the first time)
   */
  private Tab createFixedTab(Person owner) {
    Tab tab = new Tab();
    tab.setName(Tab.FIXED_TAB_NAME);
    tab.setOwner(owner.getId());
    tab.setFavorite(true);
    tab.setOrder(1);

    List<GadgetDefinition> fixedTabGadgets = gadgetDefinitionService.findAllFixedTabGadgets();
    if (CollectionUtils.isEmpty(fixedTabGadgets)) {
      //not one fixedTab gadgetDefinition, must be some local/dev environment
      return null;
    }

    int i = 0;
    for (GadgetDefinition definition : fixedTabGadgets) {
      Gadget gadget = new Gadget();
      gadget.setHasPermission(true);
      gadget.setDefinition(definition);
      gadget.setOrder(i++);
      gadget.setColumn((i % 2 == 0) ? 1 : 2);
      tab.addGadget(gadget);
    }
    tabService.saveOrUpdate(tab);
    return tab;

  }

  /*
   * Check to see if this person has the Fixed tab with some standard gadgets
   */
  private boolean containsFixedTab(List<Tab> favoriteTabList) {
    boolean result = false;
    if (!CollectionUtils.isEmpty(favoriteTabList)) {
      for (Tab tab : favoriteTabList) {
        if (tab.isFixedTab()) {
          result = true;
          break;
        }
      }

    }
    return result;
  }

  private void addTeamToModelMap(ModelMap modelMap, Tab tab, GroupList groupList) {
    if (StringUtils.hasText(tab.getTeam())) {
      Group group = groupList.getGroupById(tab.getTeam());

      modelMap.addAttribute("team", group);
    }
  }

  /*
   * We going to query and see how many updates and invites this user has
   */
  private void addInvitesToModelMap(ModelMap modelMap, Person owner) {
    // see ${nbrInvites} in genericpage.tag
    int sharedInvites = inviteService.getCountRecentByInvitee(owner.getId(),
            InviteStatus.OPEN);
    modelMap.addAttribute("nbrInvites", sharedInvites);
  }

  /*
   * Add the gadgets of the users' tab to the modelmap
   */
  private void addGadgetsToModelMap(ModelMap modelMap, Tab tab, Person owner)
          throws IOException, BlobCrypterException {
    List<Gadget> gadgets = tab.getGadgets();

    if (!CollectionUtils.isEmpty(gadgets)) {
      modelMap.addAttribute("gadgetListColumn1",
              getGadgetsForColumn(gadgets, 1));
      modelMap.addAttribute("gadgetListColumn2",
              getGadgetsForColumn(gadgets, 2));
    }

    modelMap.addAttribute("gadgetSecurityTokens",
            getSecurityTokens(gadgets, owner.getId()));
    modelMap.addAttribute("gadgetGroupAwareness", getGroupAwareness(gadgets));

    String metadata = metadataProvider.getMetaData(gadgets);
    modelMap.addAttribute("metadata", metadata);
  }

  protected String getGroupAwareness(List<Gadget> gadgets) {
    String list = "[";
    for (int i = 0; i < gadgets.size(); i++) {
      if (i > 0) {
        list += ",";
      }
      Gadget g = gadgets.get(i);

      list += g.getDefinition().isSupportsGroups();
    }
    return list + "]";
  }

  protected String getSecurityTokens(List<Gadget> gadgets, String personId)
          throws BlobCrypterException {
    String tokens = "[";

    // TODO get this from blobkey.txt?
    BasicBlobCrypter blobCrypter = new BasicBlobCrypter(
            "nvOznUZ/XTq5sk9heC58pMeUqRM5IHH9wKcTc4UxKGc=".getBytes());

    for (int i = 0; i < gadgets.size(); i++) {
      if (i > 0) {
        tokens += ",";
      }
      Gadget g = gadgets.get(i);

      BlobCrypterSecurityToken st = new BlobCrypterSecurityToken(blobCrypter,
              environment.getContainerName(), "localhost"); // TODO fix parameters?
      st.setViewerId(personId);
      st.setOwnerId(personId);
      st.setAppUrl(g.getDefinition().getUrl());
      st.setGroupContext(g.getTab().getTeam());
      String token = Utf8UrlCoder.encode(st.encrypt());

      tokens += "'" + token + "'";
    }
    return tokens + "]";
  }

  final Tab getRequestedTab(HttpServletRequest request, Person owner, List<Tab> favoriteTabList) {
    String tabId = request.getParameter("tab");
    Tab tab = null;
    if (StringUtils.hasText(tabId)) {
      tab = tabService.findByIdAndOwner(Long.parseLong(tabId), owner.getId());
    }
    if (tab != null) {
      return tab;
    } else if (!CollectionUtils.isEmpty(favoriteTabList)) {
      return favoriteTabList.get(0);
    }
    return null;
  }

  private List<Gadget> getGadgetsForColumn(List<Gadget> gadgets, int column) {
    List<Gadget> result = new ArrayList<Gadget>();
    for (Gadget gadget : gadgets) {
      if (gadget.getColumn() == column) {
        result.add(gadget);
      }
    }
    return result;
  }

  public void setCoinEnvironment(CoinEnvironment env) {
    this.environment = env;
  }
}
