/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package nl.surfnet.coin.portal.control;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;

import javax.servlet.http.HttpServletRequest;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;

import nl.surfnet.coin.portal.domain.IdentityProvider;
import nl.surfnet.coin.portal.service.IdentityProviderService;
import nl.surfnet.coin.portal.service.TextContentService;
import nl.surfnet.coin.portal.util.CoinEnvironment;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.LocaleResolver;
import org.xml.sax.SAXException;

import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.fetcher.FeedFetcher;
import com.sun.syndication.fetcher.FetcherException;
import com.sun.syndication.fetcher.impl.FeedFetcherCache;
import com.sun.syndication.fetcher.impl.HashMapFeedInfoCache;
import com.sun.syndication.fetcher.impl.HttpURLFeedFetcher;
import com.sun.syndication.io.FeedException;

/**
 * {@link Controller} that handles the landing page (e.g. home page of the
 * coin-portal).
 */
@Controller
public class LandingPageController {
  private static final String IDP_EMPTY_LOGO = "media/transparant.png";

  private Logger logger = Logger.getLogger(this.getClass());

  private static final String OTHER = "OTHER";

  private static final String NL = "NL";

  private static final Comparator<? super IdentityProvider> IDENTITY_PROVIDER_COMPARATOR = new IdentityProviderComparator();

  @Autowired
  private CoinEnvironment environment;

  @Autowired
  private LocaleResolver localeResolver;

  @Autowired
  private IdentityProviderService identityProviderService;

  @Autowired
  private TextContentService textContentService;

  @RequestMapping("/landingpage.shtml")
  public String start(ModelMap modelMap, HttpServletRequest request)
      throws XPathExpressionException, IOException, SAXException,
      ParserConfigurationException, NoSuchAlgorithmException, JSONException {
    modelMap.addAttribute("environment", environment);

    addIdps(modelMap, request);
    addFeeds(modelMap);
    addTextContent(modelMap);

    return "landingpage";
  }

  private void addTextContent(ModelMap modelMap) {
    modelMap.addAttribute("textContentIntroduction", textContentService
        .getByField(environment.getTextFieldIntroduction()).getText());
    modelMap.addAttribute("textContentGettingStarted", textContentService
        .getByField(environment.getTextFieldGettingStarted()).getText());
  }

  /**
   * If URLs for feeds are configured in {@link #environment} (as comma
   * separated string) their feeds are fetched and added to the {@link ModelMap}
   * .
   * 
   * @param modelMap
   *          The {@link ModelMap} for this Controller
   */
  void addFeeds(ModelMap modelMap) {
    String feedLocationProp = environment.getLandingPageRssFeeds();
    if (StringUtils.isBlank(feedLocationProp)) {
      return;
    }
    // Use Apache Commons StringUtils, the one from SpringFramework returns null
    // if the String to split does not contain a ',' instead of an array with 1
    // item
    String[] locations = StringUtils.split(feedLocationProp, ',');
    modelMap.addAttribute("feeds", getFeeds(locations));
  }

  private void addIdps(ModelMap modelMap, HttpServletRequest request)
      throws NoSuchAlgorithmException, UnsupportedEncodingException,
      JSONException {
    List<IdentityProvider> idps = identityProviderService.getIdps();

    Collections.sort(idps, IDENTITY_PROVIDER_COMPARATOR);

    String idpsString = prepareIdps(idps, localeResolver.resolveLocale(request));

    modelMap.addAttribute("idps", idpsString);
  }

  private String prepareIdps(List<IdentityProvider> idps, Locale locale)
      throws JSONException, NoSuchAlgorithmException,
      UnsupportedEncodingException {
    JSONArray idpsJson = new JSONArray();
    for (IdentityProvider idp : idps) {
      String logo = idp.getLogo() != null ? idp.getLogo() : IDP_EMPTY_LOGO;

      JSONObject idpJson = new JSONObject();
      idpJson.put("ID", getMD5(idp.getEntityId()));
      idpJson.put("Name", idp.getDisplayName(locale.getLanguage()));
      idpJson.put("Logo", logo);
      idpJson.put("EntityId", idp.getEntityId());
      idpJson.put("Keywords", idp.getKeywords(locale.getLanguage()));

      idpsJson.put(idpJson);
    }

    return idpsJson.toString();
  }

  private String getMD5(String string) throws NoSuchAlgorithmException,
      UnsupportedEncodingException {
    byte[] bytesOfString = string.getBytes("UTF-8");
    MessageDigest md = MessageDigest.getInstance("MD5");
    byte[] digest = md.digest(bytesOfString);
    BigInteger bigInt = new BigInteger(1, digest);
    String hashtext = bigInt.toString(16);

    return hashtext;
  }

  /**
   * Gets a List of (cached) {@link com.sun.syndication.feed.synd.SyndFeed} for
   * the given locations
   * 
   * @param feedLocations
   *          String[] of the URLs of the RSS/Atom feed
   * @return List of {@link com.sun.syndication.feed.synd.SyndFeed} for the
   *         locations, may be empty
   */
  private List<SyndFeed> getFeeds(String[] feedLocations) {
    List<SyndFeed> feeds = new ArrayList<SyndFeed>();
    for (String feedLocation : feedLocations) {
      try {
        FeedFetcherCache feedInfoCache = HashMapFeedInfoCache.getInstance();
        FeedFetcher feedFetcher = new HttpURLFeedFetcher(feedInfoCache);
        SyndFeed feed = feedFetcher.retrieveFeed(new URL(feedLocation));
        if (feed != null) {
          feeds.add(feed);
        }
      } catch (IOException e) {
        logger.warn("Connection error retrieving feed", e);
      } catch (FeedException e) {
        logger.warn("Received invalid feed which makes it impossible to parse",
            e);
      } catch (FetcherException e) {
        logger.warn("HTTP error retrieving feed", e);
      }
    }
    return feeds;
  }

  public void setEnvironment(CoinEnvironment environment) {
    this.environment = environment;
  }

  /**
   * @param textContentService the textContentService to set
   */
  public void setTextContentService(TextContentService textContentService) {
    this.textContentService = textContentService;
  }
}
