/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.control;

import nl.surfnet.coin.portal.domain.GadgetDefinition;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.portal.domain.TemplateTab;
import nl.surfnet.coin.portal.service.TemplateTabService;
import org.opensocial.models.Person;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletRequest;

import java.util.Set;

@Controller
public class TabController extends BaseController {

  @Autowired
  private TemplateTabService templateTabService;

  @RequestMapping(value = "add-tab.shtml", method = RequestMethod.POST)
  public @ResponseBody
  String add(ModelMap modelMap, HttpServletRequest request) {
    String tabName = request.getParameter("name");

    Person owner = getPerson(request);

    Tab tab = new Tab();
    tab.setName(tabName);
    tab.setOwner(owner.getId());
    tab.setFavorite(true);

    String templateId = request.getParameter("templateTab");
    if (StringUtils.hasText(templateId)) {
      TemplateTab templateTab = templateTabService.findById(Long
          .parseLong(templateId));
      Set<GadgetDefinition> gadgetDefinitions = templateTab
          .getGadgetDefinitions();
      for (GadgetDefinition gadgetDefinition : gadgetDefinitions) {
        tab.addGadget(gadgetDefinition.protoType());
      }
    }

    tab.setOrder(tabService.getCountByOwner(owner.getId(), true) + 1);

    tab.setId(tabService.saveOrUpdate(tab));

    return tab.getId().toString();
  }

  @RequestMapping(value = "rename-tab.shtml", method = RequestMethod.POST)
  public @ResponseBody
  boolean rename(ModelMap modelMap, HttpServletRequest request) {
    String name = request.getParameter("name");
    if (name == null || name.isEmpty()) {
      logger.debug("Rename tab failed: no new name for tab");
      return false;
    }
    Tab tab = getTab(request, request.getParameter("id"));
    if (tab == null) {
      logger.debug("Rename tab failed: couldn't get the tab");
      return false;
    }
    tab.setName(name);
    tabService.saveOrUpdate(tab);
    return true;
  }

  @RequestMapping(value = "remove-tab.shtml", method = RequestMethod.POST)
  public @ResponseBody
  boolean remove(ModelMap modelMap, HttpServletRequest request) {
    Tab tab = getTab(request, request.getParameter("id"));
    if (tab == null) {
      logger.info("Removal of tab failed, couldn't get the tab");
      return false;
    }
    tabService.delete(tab);
    return true;
  }

  @RequestMapping(value = "reorder-tabs-sidebar.shtml", method = RequestMethod.POST)
  public @ResponseBody
  boolean reorder(ModelMap modelMap, HttpServletRequest request) {
    String[] newOrder = request.getParameterValues("ids[]");
    Person owner = getPerson(request);
    return tabService.reorderTabs(newOrder, owner.getId(), true);
  }

  @RequestMapping(value = "unset-favorite-tab.shtml", method = RequestMethod.POST)
  public @ResponseBody
  boolean unsetFavoriteTab(ModelMap modelMap, HttpServletRequest request) {
    boolean favorite = Boolean.valueOf(request.getParameter("favorite"));
    Tab tab = getTab(request, request.getParameter("id"));
    if (tab == null) {
      logger.info("Could not retrieve the tab for setting the favorite");
      return false;
    }
    tab.setFavorite(Boolean.valueOf(favorite));
    tabService.saveOrUpdate(tab);
    return true;
  }

  @RequestMapping(value = "set-tab-team.shtml", method = RequestMethod.POST)
  public @ResponseBody
  boolean setTabTeam(ModelMap modelMap, HttpServletRequest request) {
    String teamName = request.getParameter("team");
    String teamTitle = request.getParameter("teamTitle");
    Tab tab = getTab(request, request.getParameter("id"));
    if (tab == null || teamName == null || !isMemberOfGroup(request, teamName)) {
      logger.info("Could not set the tab to a team");
      return false;
    }
    tab.setTeam(teamName);
    tab.setTeamTitle(teamTitle);
    tabService.saveOrUpdate(tab);
    return true;

  }

  @RequestMapping(value = "unset-tab-team.shtml", method = RequestMethod.POST)
  public @ResponseBody
  boolean unsetTabTeam(ModelMap modelMap, HttpServletRequest request) {
    Tab tab = getTab(request, request.getParameter("id"));
    if (tab == null) {
      logger.info("Could not resolve the tab");
      return false;
    }
    tab.setTeam(null);
    tab.setTeamTitle(null);
    tabService.saveOrUpdate(tab);
    return true;
  }

}
