/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.domain;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.springframework.util.StringUtils;

public class IdentityProvider {
  
  private static final String DELIMITER = ",";

  private String entityId;

  private Map<String, String> displayNames;

  private String displayName;

  private String logo;

  private String geoLocationHint;

  private Locale defaultLocale;
  
  private String country;
  
  private Map<String, String[]> keywords;

  public IdentityProvider() {
    entityId = null;
    displayNames = new HashMap<String, String>();
    displayName = null;
    defaultLocale = null;
    country = null;
    keywords = new HashMap<String, String[]>();
  }

  /**
   * Set a display name.
   * 
   * @param language
   *          the language for the display name to set
   * @param displayName
   *          the display name to set
   */
  public void setDisplayName(String language, String displayName) {
    displayNames.put(language, displayName);
  }

  /**
   * Get the display name for a given locale.
   * 
   * @param language
   * @return the display name for the given locale or the display name for the
   *         default locale, or null if no default locale is set return null
   */
  public String getDisplayName(String language) {
    return displayNames.containsKey(language) ? displayNames.get(language)
        : defaultLocale != null ? displayNames.containsKey(defaultLocale.getLanguage()) ? displayNames
            .get(defaultLocale.getLanguage()) : null
            : null;
  }

  /**
   * Set the entityID.
   * 
   * @param entityId
   *          the entityId to set
   */
  public void setEntityId(String entityId) {
    this.entityId = entityId;
  }

  /**
   * Get the entityID.
   * 
   * @return the entityId
   */
  public String getEntityId() {
    return entityId;
  }

  /**
   * Set the default Locale.
   * 
   * @param defaultLocale
   *          the defaultLocale to set
   */
  public void setDefaultLocale(Locale defaultLocale) {
    this.defaultLocale = defaultLocale;
  }

  /**
   * Get the default locale.
   * 
   * @return the defaultLocale
   */
  public Locale getDefaultLocale() {
    return defaultLocale;
  }

  /**
   * Set the logo.
   * 
   * @param logo
   *          the logo to set
   */
  public void setLogo(String logo) {
    this.logo = logo;
  }

  /**
   * Get the logo
   * 
   * @return the logo
   */
  public String getLogo() {
    return logo;
  }

  /**
   * Set the geolocation and also set the country based on the geolocation.
   * The country will be set to NL or OTHER
   * 
   * @param geoLocationHint
   *          the geoLocationHint to set
   */
  public void setGeoLocationHint(String geoLocationHint) {
    this.geoLocationHint = geoLocationHint;
    
    String[] geoLoc = StringUtils.split(geoLocationHint, DELIMITER);
    
    Double longitude = 0.0;
    Double latitude = 0.0;

    if (!StringUtils.hasText(geoLoc[0]) || !StringUtils.hasText(geoLoc[1]) ) {
      longitude = new Double(geoLoc[0]);
      latitude = new Double(geoLoc[1]);
      
      // The Geolocation of NL is between 51 - 53 longitude and 4 - 7 latitude
      setCountry(((longitude < 51 || longitude > 53) && (latitude < 4 && latitude > 7)) ? "OTHER" : "NL");
    } else {
      // set the country to NL if the Geolocation is empty.
      setCountry("NL");
    }
  }

  /**
   * Get the GEO location hint
   * 
   * @return the geoLocationHint
   */
  public String getGeoLocationHint() {
    return geoLocationHint;
  }

  /**
   * Set the display name, used by controllers to set the display name for the
   * users' locale.
   * 
   * @param displayName
   *          the displayName to set
   */
  public void setDisplayName(String displayName) {
    this.displayName = displayName;
  }

  /**
   * Get the display name.
   * 
   * @return the displayName
   */
  public String getDisplayName() {
    if (displayName != null) {
      return displayName;
    } else {
      return defaultLocale != null && displayNames.containsKey(defaultLocale.getLanguage()) ? displayNames.get(defaultLocale.getLanguage()) : null;
    }
  }

  /**
   * Set the country.
   * 
   * @param country the country to set
   */
  public void setCountry(String country) {
    this.country = country;
  }

  /**
   * Get the country.
   * 
   * @return the country
   */
  public String getCountry() {
    return country;
  }

  /**
   * @param keywords the keywords to set
   */
  public void setKeywords(Map<String, String[]> keywords) {
    this.keywords = keywords;
  }

  /**
   * @return the keywords
   */
  public Map<String, String[]> getKeywords() {
    return keywords;
  }

  public void setKeyword(String language, String[] keywords) {
    this.keywords.put(language, keywords);
  }
  
  public String[] getKeywords(String language) {
    return keywords.containsKey(language) ? keywords.get(language)
        : defaultLocale != null ? keywords.containsKey(defaultLocale.getLanguage()) ? keywords
            .get(defaultLocale.getLanguage()) : null
            : null;
  }

}
