/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.interceptor;

import nl.surfnet.coin.opensocial.service.PersonService;
import nl.surfnet.coin.portal.util.CoinEnvironment;
import nl.surfnet.coin.shared.domain.ErrorMail;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import org.opensocial.models.Person;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import java.net.InetAddress;
import java.net.UnknownHostException;

/**
 * Intercepts calls to controllers to handle Single Sign On details from
 * Shibboleth and sets a Person object on the session when the user is logged
 * in.
 */
public class LoginInterceptor extends HandlerInterceptorAdapter {

  public static final String PERSON_SESSION_KEY = "person";
  public static final String USER_STATUS_SESSION_KEY = "userStatus";
  public static final String INVITEES = "invitees";

  @Autowired
  @Qualifier("opensocialPersonService")
  private PersonService personService;

  @Autowired
  private CoinEnvironment coinEnvironment;

  @Autowired
  private ErrorMessageMailer errorMessageMailer;

  @Override
  public boolean preHandle(HttpServletRequest request,
                           HttpServletResponse response, Object handler) throws Exception {

    HttpSession session = request.getSession();

    String remoteUser = getRemoteUser(request);

    // Check session first:
    Person person = (Person) session.getAttribute(PERSON_SESSION_KEY);
    if (person == null || !person.getId().equals(remoteUser)) {

      if (StringUtils.hasText(remoteUser)) {

        // Lookup person and add to session:
        person = personService.getPerson(remoteUser, remoteUser);
        session.setAttribute(PERSON_SESSION_KEY, person);

        // Add the user status to the session
        String userStatus = getUserStatus(request);
        userStatus = StringUtils.hasText(userStatus) ? userStatus : "guest";

        session.setAttribute(USER_STATUS_SESSION_KEY, userStatus);

        if (person == null) {
          sendErrorMail(remoteUser);
        }

        // User is not logged in, and REMOTE_USER header is empty.
        // Check whether the user is requesting the landing page, if not
        // redirect him to the landing page.
      } else {
        String url = request.getRequestURI();
        String[] urlSplit = url.split("/");

        // Unprotect the javascript files and media files
        if (urlSplit[2].equals("js") || urlSplit[2].equals("media")) {
          return super.preHandle(request, response, handler);
        } else if (!url.equals("/coin/landingpage.shtml")) {
          response.sendRedirect("/coin");
          return false;
        }
      }
    }
    return super.preHandle(request, response, handler);
  }

  private void sendErrorMail(String remoteUser) throws ServletException {
    String shortMessage = "Cannot find user: " + remoteUser;
    String errorMessage = "An exception occured in the Conext portal LoginInterceptor.<br/><br/>" +
            getClass().getSimpleName() + "<br/>" + shortMessage;
    ErrorMail errorMailDetails = new ErrorMail(shortMessage, errorMessage, remoteUser, "UNKNOWN", getHost(), "Portal");
    errorMessageMailer.sendErrorMail(errorMailDetails);
    throw new ServletException(shortMessage);
  }

  private String getHost() {
    try {
      return InetAddress.getLocalHost().toString();
    } catch (UnknownHostException e) {
      return "UNKNOWN";
    }
  }

  /**
   * Hook for subclasses to override the shibboleth default behaviour
   *
   * @param request the httpRequest
   * @return the String of the logged in user
   */
  protected String getRemoteUser(HttpServletRequest request) {
    return request.getHeader("REMOTE_USER");
  }

  /**
   * Hook for subclasses to override the shibboleth default behaviour
   *
   * @param request the httpRequest
   * @return the String of the user status
   */
  protected String getUserStatus(HttpServletRequest request) {
    return request.getHeader("coin-user-status");
  }

  /**
   * @param personService the personService to set
   */
  public void setPersonService(PersonService personService) {
    this.personService = personService;
  }

  /**
   * @param coinEnvironment the coinEnvironment to set
   */
  public void setCoinEnvironment(CoinEnvironment coinEnvironment) {
    this.coinEnvironment = coinEnvironment;
  }

  /**
   * @return the coinEnvironment
   */
  protected CoinEnvironment getCoinEnvironment() {
    return coinEnvironment;
  }


  public void setErrorMessageMailer(ErrorMessageMailer errorMessageMailer) {
    this.errorMessageMailer = errorMessageMailer;
  }
}
