/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.service;

import nl.surfnet.coin.portal.domain.ClonedTab;
import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.Tab;
import nl.surfnet.coin.shared.service.GenericService;

import java.util.List;

/**
 * Service for managing tabs
 */
public interface TabService extends GenericService<Tab> {

  /**
   * Find favorites.
   *
   * @param ownerId   the {@link String} with the Id of the owner
   * @param favorites true for all favorites, false for non-favorites.
   * @return Tabs
   */
  List<Tab> findFavorites(String ownerId, boolean favorites);

  /**
   * Reorder favorite tabs.
   *
   * @param newOrder the {@link String[]} containing the new order
   * @param ownerId  the {@link String} with the Id of the owner
   * @param {@link boolean} saying whether these tabs are favorites or not.
   * @return <code>true</code> if the reordering was succesful <code>false</code> if it wasn't
   */
  boolean reorderTabs(String[] newOrder, String ownerId, boolean favorites);

  /**
   * Get a count of all tabs.
   *
   * @param ownerId   the {@link String} with the Id of the owner
   * @param favorites boolean indicating the favorite nature.
   * @return {@link int} the number of all tabs by the given owner
   */
  int getCountByOwner(String ownerId, boolean favorites);

  /**
   * Clone a tab.
   *
   * @param tab the tab to clone.
   * @param sharedGadgets the {@link List} containing the {@link Gadget}s
   * @return the {@link ClonedTab}
   */
  ClonedTab cloneTab(Tab tab, List<Gadget> sharedGadgets);

  /**
   * Get all the tabs of a person.
   *
   * @param ownerId@return
   * @return a {@link List} of all {@link Gadget}s
   */
  List<Tab> getTabsByOwner(String ownerId);

  /**
   * Finds a tab by Id and owner
   *
   * @param id      of the {@link nl.surfnet.coin.portal.domain.Tab}
   * @param ownerId the {@link String} with the Id of the owner
   * @return Tab or {@link null} if it can't be found
   */
  Tab findByIdAndOwner(Long id, String ownerId);
}
