/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.service.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.domain.GadgetDefinition;
import nl.surfnet.coin.portal.model.GadgetSpecWrapper;
import nl.surfnet.coin.portal.model.JsonGadgetSpec;
import nl.surfnet.coin.portal.service.GadgetDefinitionBuilder;
import nl.surfnet.coin.portal.service.MetadataProvider;

import org.apache.log4j.Logger;
import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.util.StringUtils;

/**
 * Class to build up a {@link nl.surfnet.coin.portal.domain.GadgetDefinition} from an external location
 */
public class GadgetDefinitionBuilderImpl implements GadgetDefinitionBuilder{

  private Logger logger = Logger.getLogger(this.getClass());

  private MetadataProvider metadataProvider;

  public GadgetDefinitionBuilderImpl(MetadataProvider metadataProvider) {
    this.metadataProvider = metadataProvider;

  }

  /**
   * {@inheritDoc}
   */
  public GadgetDefinition build(String gadgetUrl) {
    String metaData = getGadgetMetadata(gadgetUrl);
    JsonGadgetSpec gadgetSpec = getGadgetSpec(metaData);
    if (gadgetSpec == null || gadgetSpec.containsErrors()) {
      return null;
    }
    return fillGadgetDefinitionFromSpec(gadgetSpec, gadgetUrl);
  }

  /**
   * Returns {@link JsonGadgetSpec} for the gadget moduleprefs based on the gadgetdefinition location
   *
   * @param metaDataString {@link String} with the metadata
   * @return {@link JsonGadgetSpec} with the gadget moduleprefs
   */
  JsonGadgetSpec getGadgetSpec(String metaDataString) {
    ObjectMapper objectMapper = new ObjectMapper();
    GadgetSpecWrapper gadgetSpecWrapper = null;
    try {
      gadgetSpecWrapper = objectMapper.readValue(metaDataString, GadgetSpecWrapper.class);
    } catch (IOException e) {
      logger.info(String.format("Could not parse input as Json object"), e);
    }
    if (gadgetSpecWrapper == null || gadgetSpecWrapper.getGadgets().size() == 0) {
      return null;
    }
    return gadgetSpecWrapper.getGadgets().get(0);
  }

  /**
   * Gets the metadata of a gadget as String (serialized Json)
   * @param gadgetUrl Location of the gadget definition
   * @return String that represents the gadget metadata (as Json)
   */
  String getGadgetMetadata(String gadgetUrl) {
    long now = new Date().getTime();
    GadgetDefinition def = new GadgetDefinition();
    def.setUrl(gadgetUrl);
    def.setId(now);
    Gadget gadget = new Gadget();
    gadget.setId(now);
    gadget.setDefinition(def);
    List<Gadget> gadgets = new ArrayList<Gadget>(1);
    gadgets.add(gadget);
    return metadataProvider.getMetaData(gadgets);
  }

  /**
   * Creates a {@link GadgetDefinition} and fills it with the values
   *
   * @param gadgetSpec
   * @param gadgetUrl  identifies the gadget and is used in case the modulePrefs does not contain a title
   * @return {@link GadgetDefinition} with values from the modulePrefs.
   */
  GadgetDefinition fillGadgetDefinitionFromSpec(JsonGadgetSpec gadgetSpec, String gadgetUrl) {
    GadgetDefinition gadgetDefinition;
    gadgetDefinition = new GadgetDefinition();

    String title = StringUtils.hasText(gadgetSpec.getTitle()) ? gadgetSpec.getTitle() : gadgetUrl;
    gadgetDefinition.setTitle(title);

    gadgetDefinition.setUrl(gadgetUrl);
    gadgetDefinition.setCustomGadget(true);
    gadgetDefinition.setAdded(new Date());
    gadgetDefinition.setAuthor(gadgetSpec.getAuthor());
    gadgetDefinition.setAuthorEmail(gadgetSpec.getAuthorEmail());
    gadgetDefinition.setDescription(gadgetSpec.getDescription());
    if (gadgetSpec.getScreenshot() != null) {
      gadgetDefinition.setScreenshot(gadgetSpec.getScreenshot());
    }
    if (gadgetSpec.getThumbnail() != null) {
      gadgetDefinition.setThumbnail(gadgetSpec.getThumbnail());
    }
    return gadgetDefinition;
  }

}
