/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package nl.surfnet.coin.portal.service.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.List;

import nl.surfnet.coin.portal.domain.Gadget;
import nl.surfnet.coin.portal.service.MetadataProvider;
import nl.surfnet.coin.portal.util.CoinEnvironment;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.scheme.PlainSocketFactory;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.tsccm.ThreadSafeClientConnManager;
import org.codehaus.jackson.JsonGenerationException;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.node.ArrayNode;
import org.codehaus.jackson.node.ObjectNode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * {@link MetadataProvider} based on the shindig container
 * 
 */
@Component("metadataProvider")
public class MetadataProviderImpl implements MetadataProvider {

  @Autowired
  private CoinEnvironment environment;
  private DefaultHttpClient httpClient;

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.portal.service.MetadataProvider#getMetaData(java.util.List)
   */
  @Override
  public String getMetaData(List<Gadget> gadgets) {
    try {
      return doGetMetadata(gadgets);
    } catch (Exception e) {
      throw new RuntimeException("Exception in retrieving the metadata", e);
    }
  }

  private String doGetMetadata(List<Gadget> gadgets)
      throws Exception {
    // Prepare the JSON:
    ObjectMapper mapper = new ObjectMapper();
    ObjectNode data = mapper.createObjectNode();
    ObjectNode context = data.putObject("context");
    context.put("country", "default");
    context.put("language", "default");
    context.put("view", "home"); // default
    context.put("container", environment.getContainerName()); // default
    ArrayNode gadgetsArray = data.putArray("gadgets");

    if (gadgets != null) {
      for (Gadget gadget : gadgets) {
        ObjectNode gadgetNode = mapper.createObjectNode();
        gadgetNode.put("url", gadget.getDefinition().getUrl());
        gadgetNode.put("moduleId", "" + gadget.getId());
        gadgetsArray.add(gadgetNode);
      }
    }

    // Make the POST:
    HttpClient client = getHttpClient();
    HttpPost httpPost = new HttpPost(environment.getMetadataUrl());
    HttpEntity entity = new StringEntity(mapper.writeValueAsString(data),
        "application/javascript", null);
    httpPost.setEntity(entity);

    HttpResponse response = client.execute(httpPost);
    int statusCode = response.getStatusLine().getStatusCode();
    if (statusCode != HttpStatus.SC_OK) {
      throw new IOException("statusCode = " + statusCode);
    }

    InputStream content = response.getEntity().getContent();
    OutputStream output = new ByteArrayOutputStream();
    IOUtils.copy(content, output);
    content.close();
    return output.toString();
  }

  private HttpClient getHttpClient() throws KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException {
    if (this.httpClient != null) {
      return httpClient;
    }

    /*
     * Initialise the schemaRegistry for connecting to Engine Block
     */
    /*
     * See for documentation
     * http://java.sun.com/javase/6/docs/technotes/guides/security
     * /jsse/JSSERefGuide.html
     */
    SchemeRegistry schemeRegistry = new SchemeRegistry();
    /*
     * Only used for internal https communication (no need for certificate
     * validation)
     */
    schemeRegistry.register(new Scheme("https", 443, new SSLSocketFactory(
        new TrustStrategy() {
          @Override
          public boolean isTrusted(X509Certificate[] chain, String authType)
              throws CertificateException {
            return true;
          }
        })));
    schemeRegistry.register(new Scheme("http", 80, PlainSocketFactory
        .getSocketFactory()));
    /*
     * To re-use connections we use the ThreadSafeClientConnManager
     */
    ThreadSafeClientConnManager connManager = new ThreadSafeClientConnManager(
        schemeRegistry);
    connManager.setMaxTotal(50);
    // There is only one route, so the maximum per route equals the maximum
    // total
    connManager.setDefaultMaxPerRoute(50);
    this.httpClient = new DefaultHttpClient(connManager);

    return this.httpClient;
  }

  /**
   * @param environment
   *          the environment to set
   */
  protected void setEnvironment(CoinEnvironment environment) {
    this.environment = environment;
  }

}
