/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package nl.surfnet.coin.portal.servlet;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.conn.scheme.PlainSocketFactory;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.tsccm.ThreadSafeClientConnManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

/**
 * Proxy servlet that accepts incoming http request (from GModules),
 * gets the result from the target https server and responds the result
 * to GModules.
 */
public class GadgetDefinitionProxyServlet extends HttpServlet {
  private DefaultHttpClient httpClient;
  private ThreadSafeClientConnManager connectionManager;
  private static final Logger logger =
      LoggerFactory.getLogger(GadgetDefinitionProxyServlet.class);
  /**
   * {@inheritDoc}
   */
  @Override
  public void init(ServletConfig config) throws ServletException {
    super.init(config);
    try {
      configureHttpClient();
    } catch (NoSuchAlgorithmException e) {
      throw new ServletException("Error setting up secure httpclient", e);
    } catch (KeyManagementException e) {
      throw new ServletException("Error setting up secure httpclient", e);
    } catch (KeyStoreException e) {
      throw new ServletException("Error setting up secure httpclient", e);
    } catch (UnrecoverableKeyException e) {
      throw new ServletException("Error setting up secure httpclient", e);
    }
  }

  /**
   * Gets the destination form the contextpath, goes to the destination
   * and returns the response
   * {@inheritDoc}
   */
  @Override
  protected void doGet(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    String destination = getDestination(req, resp);
    if (destination == null) {
      return;
    }

    HttpGet externalGet = new HttpGet();
    try {
      externalGet.setURI(new URI(destination));
    } catch (URISyntaxException e) {
      throw new ServletException("Cannot convert destination '" + destination + "' to URI", e);
    }

    final HttpResponse externalResponse = httpClient.execute(externalGet);
    final int statuscode = externalResponse.getStatusLine().getStatusCode();
    resp.setStatus(statuscode);
    if (logger.isDebugEnabled()) {
      logger.debug("GadgetDefinition: {} response code: {}", destination, statuscode);
    }

    Header contentTypeHeader = externalResponse.getFirstHeader("Content-Type");
    if (contentTypeHeader != null && StringUtils.hasText(contentTypeHeader.getValue())) {
      resp.setHeader(contentTypeHeader.getName(), contentTypeHeader.getValue());
    }
    IOUtils.copy(externalResponse.getEntity().getContent(), resp.getOutputStream());
  }

  /**
   * Returns the destination that contains the gadget definition xml
   *
   * @param req  incoming {@link HttpServletRequest}
   * @param resp outgoind {@link HttpServletResponse}
   * @return gadget definition location (https://www.example.com/gadgetdefinition.xml)
   * @throws ServletException if the request does not contain enough information
   */
  String getDestination(HttpServletRequest req, HttpServletResponse resp) throws ServletException {
    String reqPathInfo = req.getPathInfo();

    if (reqPathInfo == null || (reqPathInfo.trim().length() <= "/.xml".length())) {
      try {
        resp.sendError(HttpServletResponse.SC_BAD_REQUEST,
            "No destination found in the context path");
        return null;
      } catch (IOException e) {
        throw new ServletException("No request params AND could not write error message to response", e);
      }
    }
    return new String(Base64.decodeBase64(reqPathInfo.substring(1, reqPathInfo.indexOf(".xml"))));
  }

  @Override
  public void destroy() {
    if (connectionManager != null) {
      connectionManager.shutdown();
    }
    super.destroy();
  }

  private void configureHttpClient() throws NoSuchAlgorithmException,
      KeyManagementException, KeyStoreException, UnrecoverableKeyException {
    /*
     * See for documentation
     * http://java.sun.com/javase/6/docs/technotes/guides/security/jsse/JSSERefGuide.html
     */
    SchemeRegistry schemeRegistry = new SchemeRegistry();
    /*
     * We trust everyone
     */
    schemeRegistry.register(new Scheme("https", 443, new SSLSocketFactory(
        new TrustStrategy() {
          @Override
          public boolean isTrusted(X509Certificate[] chain, String authType)
              throws CertificateException {
            return true;
          }
        })));
    schemeRegistry.register(new Scheme("http", 80, PlainSocketFactory
        .getSocketFactory()));
    /*
     * To re-use connections we use the ThreadSafeClientConnManager
     */
    connectionManager = new ThreadSafeClientConnManager(
        schemeRegistry);
    connectionManager.setMaxTotal(50);
    // There is only one route, so the maximum per route equals the maximum
    // total
    connectionManager.setDefaultMaxPerRoute(50);
    httpClient = new DefaultHttpClient(connectionManager);
  }
}
