/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
* 
* http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/


package nl.surfnet.coin.portal.servlet;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Properties;

/**
 * Servlet that fetches the gadgetsettings from GModules.
 * <p/>
 * Example usage: http[s]://[host]:[port]/[context]/[servletname]?mid=&output=&up_groupContext=&url=
 */
public class GadgetSettingsProxyServlet extends HttpServlet {
  private static final Logger logger =
      LoggerFactory.getLogger(GadgetSettingsProxyServlet.class);
  private static final String G_MODULES_URL_PARAM = "gModulesUrl";
  private static final String URL_PATTERN = "https?://([-\\w\\.]+)+(:\\d+)?(/([\\w/_\\.]*(\\?\\S+)?)?)?";
  private static final String PARAM_URL = "url";
  private static final String UTF_8 = "UTF-8";

  private String gModulesUrl;
  private String gadgetProxyUrl;
  private HttpClient httpClient;
  private MultiThreadedHttpConnectionManager connectionManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public void init(ServletConfig config) throws ServletException {
    super.init(config);
    configureGModulesUrl(config);
    configureHttpClient();
    configureGadgetProxyUrl();
  }

  /**
   * Passes all query parameters to GModules and returns the response given by GModules.
   * {@inheritDoc}
   */
  @Override
  protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException {
    String qString = getQueryString(req, resp);
    if (qString == null) {
      return;
    }

    final String gModulesUrl = getGModulesUrl();
    GetMethod getMethod = new GetMethod(gModulesUrl);
    getMethod.setQueryString(qString);
    getMethod.setFollowRedirects(true);

    int statusCode = executeRequest(getMethod);
    if (logger.isDebugEnabled()) {
      logger.debug(String.format("GModulesUrl: %s", gModulesUrl));
      logger.debug(String.format("QueryString: %s", qString));
      logger.debug(String.format("Status code from remote server:  %s", statusCode));
    }
    resp.setStatus(statusCode);
    final Header contentTypeHeader = getMethod.getResponseHeader("Content-Type");
    if (contentTypeHeader != null && StringUtils.hasText(contentTypeHeader.getValue())) {
      resp.setHeader(contentTypeHeader.getName(), contentTypeHeader.getValue());
    }
    copyResponseBody(resp, getMethod);
    getMethod.releaseConnection();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void destroy() {
    if (connectionManager != null) {
      connectionManager.shutdown();
    }
    super.destroy();
  }

  /**
   * Gets the query string from the request
   *
   * @param req  {@link HttpServletRequest}
   * @param resp {@link HttpServletResponse}
   * @return same query string or {@link HttpServletResponse#SC_BAD_REQUEST}
   *         if the string is empty
   * @throws ServletException if the query string is empty
   */
  String getQueryString(HttpServletRequest req, HttpServletResponse resp) throws ServletException {
    String qString = req.getQueryString();
    if (qString == null || "".equals(qString.trim())) {
      try {
        resp.sendError(HttpServletResponse.SC_BAD_REQUEST,
            "No request params, can never successfully request GModules settings");
        return null;
      } catch (IOException e) {
        throw new ServletException("No request params AND could not write error message to response", e);
      }
    }

    final Map parameterMap = req.getParameterMap();
    StringBuffer sb = new StringBuffer();

    boolean first = true;
    for (Object paramKey : parameterMap.keySet()) {
      if (!first) {
        sb.append('&');
      }

      Object parameterValue = parameterMap.get(paramKey);
      if (parameterValue instanceof String[]) {
        String[] values = (String[]) parameterValue;
        boolean firstValue = true;
        for (String arrValue : values) {
          if (!firstValue) {
            sb.append('&');
          }
          appendRequestParameter(sb, paramKey, arrValue);
          firstValue = false;
        }
      } else {
        appendRequestParameter(sb, paramKey, parameterValue);
      }
      first = false;
    }

    return sb.toString();

  }

  private void appendRequestParameter(StringBuffer sb, Object paramKey, Object parameterValue) {
    sb.append(paramKey).append('=');
    if (PARAM_URL.equals(paramKey)) {
      sb.append(getGadgetDefinitionUrl((String) parameterValue));
    } else {
      try {
        sb.append(URLEncoder.encode((String) parameterValue, UTF_8));
      } catch (UnsupportedEncodingException e) {
        // unlikely in our setup
        throw new RuntimeException("Server does not support UTF-8 encoding");
      }
    }
  }

  private String getGadgetDefinitionUrl(String value) {
    if (!value.startsWith("https://")) {
      return value;
    }
    return gadgetProxyUrl + new String(Base64.encodeBase64URLSafe(value.getBytes())) + ".xml";
  }


  /**
   * Executes the http request and returns its status code
   *
   * @param getMethod {@link GetMethod}
   * @return status code
   * @throws ServletException if the request failed to execute
   */
  private int executeRequest(GetMethod getMethod) throws ServletException {
    try {
      return httpClient.executeMethod(getMethod);
    } catch (IOException e) {
      throw new ServletException("Could not execute GET request", e);
    }
  }

  /**
   * Copies the response body from GModules to this servlet's {@link HttpServletResponse}
   *
   * @param resp      {@link HttpServletResponse} for this servlet
   * @param getMethod that was executed to fetch the gadget settings from GModules
   * @throws ServletException if the response stream can't be copied
   */
  private void copyResponseBody(HttpServletResponse resp, GetMethod getMethod) throws ServletException {
    try {
      IOUtils.copy(getMethod.getResponseBodyAsStream(), resp.getOutputStream());
    } catch (IOException e) {
      throw new ServletException("Could not copy response body from GModules", e);
    }
  }

  /**
   * Gets GModules' base location from the servlet config
   *
   * @param config {@link ServletConfig}
   * @throws ServletException if the config is missing or invalid
   */
  private void configureGModulesUrl(ServletConfig config) throws ServletException {
    String initParameter = config.getInitParameter(G_MODULES_URL_PARAM);
    if (initParameter == null || !initParameter.matches(URL_PATTERN)) {
      throw new ServletException("Value for gModulesUrl is not a valid url");
    }
    gModulesUrl = initParameter;
  }

  private void configureHttpClient() {
    connectionManager = new MultiThreadedHttpConnectionManager();
    httpClient = new HttpClient(connectionManager);
  }

  private void configureGadgetProxyUrl() throws ServletException {

    final InputStream systemResourceAsStream =
        this.getClass().getClassLoader().getResourceAsStream("coin-portal.properties");
    Properties properties = new Properties();
    try {
      properties.load(systemResourceAsStream);
      gadgetProxyUrl = (String) properties.get("gadgetDefinitionProxy");
      if (gadgetProxyUrl == null || "".equals(gadgetProxyUrl)) {
        throw new ServletException(
            "Empty or missing property 'gadgetDefinitionProxy' in 'coin-portal.properties'");
      }
    } catch (IOException e) {
      throw new ServletException(
          "Cannot read properties file 'coin-portal.properties' from classpath", e);
    }
  }

  /**
   * Utility method for testing
   *
   * @return base location of GModules
   */
  String getGModulesUrl() {
    return gModulesUrl;
  }

  /**
   * Utility method for testing
   *
   * @return base location for http to https proxy
   */
  String getGadgetProxyUrl() {
    return gadgetProxyUrl;
  }

}
