/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * 
 */
package nl.surfnet.coin.shindig.servlet;

import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.IOUtils;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.log4j.Logger;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.social.opensocial.model.Group;
import org.apache.shindig.social.opensocial.model.Person;
import org.apache.shindig.social.opensocial.spi.GroupService;
import org.apache.shindig.social.opensocial.spi.PersonService;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.apache.shindig.social.opensocial.spi.UserId.Type;
import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.util.StringUtils;

import nl.surfnet.coin.shindig.protocol.ContextEnvironment;
import nl.surfnet.coin.shindig.spi.CoinGuiceModule;

/**
 * @author steinwelberg
 * 
 */
@SuppressWarnings("serial")
public class StatusPageServlet extends HttpServlet {

  /*
   * For parsing the JSON
   */
  private static final ObjectMapper objectMapper = new ObjectMapper();

  /*
   * The logger
   */
  private static final Logger logger = Logger
      .getLogger(StatusPageServlet.class);

  /*
   * The usage page to be displayed if no parameters are passed
   */
  private static final String USAGE = "<html><head><title>Shindig status page - usage</title></head><body><h1>Usage:</h1><p>To check the status you have to add one (or all) of the following url parameters:</p><table><thead><tr><th>Parameter</th><th>Example</th><th>Description</th></tr></thead><tbody><tr><td>person</td><td>urn:collab:person:surfguest.nl:steinwelberg</td><td>The UserID for the person to retrieve.</td></tr><tr><td>group</td><td>urn:collab:person:test.surfguest.nl:steinwelberg</td><td>The UserID for the person to retrieve the groups for.</td></tr><tr><td>oauth</td><td>http://gadgets.steinwelberg.nl/christiaanoauth.xml</td><td>The gagdet base url for the gadget to test oauth for.</td></tr></tbody></table><h1>Example:</h1><p>%s?person=john.doe&group=john.doe&oauth=http://gadgets.steinwelberg.nl/christiaanoauth.xml</p></body></html>";

  private HttpClient httpClient = new DefaultHttpClient();
  
  private boolean error;
  
  

  @Override
  protected void doGet(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException {

    String person = req.getParameter("person");
    String group = req.getParameter("group");
    String gadgetBaseUrl = req.getParameter("oauth");

    error = false;
    String output = "";

    // Check for oauth parameters && perform oauth check if parameters are
    // available
    if (gadgetBaseUrl != null && StringUtils.hasText(gadgetBaseUrl)) {
      output += checkOauth(gadgetBaseUrl);
    }

    // Check for person parameter && perform person check if parameter is
    // available
    if (person != null && StringUtils.hasText(person)) {
      if (StringUtils.hasText(output)) {
        output += ", ";
      }
      output += checkPerson(person);
    }

    // Check for group parameter && perform group check if parameter is
    // available
    if (group != null && StringUtils.hasText(group)) {
      if (StringUtils.hasText(output)) {
        output += ", ";
      }
      output += checkGroup(group);
    }

    // Check if output is still empty, no parameters filled properly -> show
    // usage page
    if (!StringUtils.hasText(output)) {
      this.error = true;
      output += String.format(USAGE, req.getRequestURL());
    }
    
    // Was there an error?
    if (error) {
      resp.setStatus(HttpStatus.SC_INTERNAL_SERVER_ERROR);
    } else {
      resp.setStatus(HttpStatus.SC_OK);
    }

    resp.setContentType("text/html");
    resp.setCharacterEncoding("UTF-8");

    try {
      resp.getOutputStream().write(output.getBytes());
    } catch (IOException e) {
      throw new ServletException("Could not write response body", e);
    }
  }

  private String checkOauth(String gadgetBaseUrl) {
    String url = getEnvironment().getBaseSocialApiUrl();
    String consumerSecret;

    /*
     * For possible logging purposed we define the json response here
     */
    String json = null;
    try {
      url += "/service/metadata?gadgeturl="
          + URLEncoder.encode(gadgetBaseUrl, "UTF-8") + "&keys="
          + URLEncoder.encode("coin:oauth:secret", "UTF-8");
      json = IOUtils.toString(executeHttpGet(url));

      @SuppressWarnings("unchecked")
      HashMap<String, Object> result = objectMapper.readValue(json,
          HashMap.class);
      consumerSecret = (String) result.get("coin:oauth:secret");
    } catch (Exception e) {

      String errorMessage = String
          .format(
              "Unexpected exception occured in retrieving Consumer while parsing (%s)",
              json);
      logger.error(errorMessage, e);
      return "oAuth Failed";
    }
    
    if (!StringUtils.hasText(consumerSecret)) {
      error = true;
    }
    
    return StringUtils.hasText(consumerSecret) ? "oAuth OK" : "oAuth Failed";
  }

  private String checkPerson(String personId) throws ServletException {
    logger.debug("Checking for Person: " + personId);

    Person person = null;
    Future<Person> personFuture = getPersonService().getPerson(
        new UserId(Type.userId, personId), null, null);
    if (personFuture != null) {
      try {
        person = personFuture.get();
      } catch (InterruptedException e) {
        throw new ServletException("Could not fetch person", e);
      } catch (ExecutionException e) {
        throw new ServletException("Could not fetch person", e);
      }
    }
    
    if (person == null || !person.getId().equals(personId)) { 
      error = true;
    }

    return person != null && person.getId().equals(personId) ? "Person OK"
        : "Person Failed";
  }

  private String checkGroup(String personId) throws ServletException {
    logger.debug("checking groups for person: " + personId);

    int groups = 0;
    Future<RestfulCollection<Group>> groupsFuture = getGroupService()
        .getGroups(new UserId(Type.userId, personId), null, null, null);

    if (groupsFuture != null) {
      try {
        groups = groupsFuture.get().getTotalResults();
      } catch (InterruptedException e) {
        throw new ServletException("Could not fetch person", e);
      } catch (ExecutionException e) {
        throw new ServletException("Could not fetch person", e);
      }
    }
    
    if (groups == 0) {
      error = true;
    }

    return groups > 0 ? "Group OK" : "Group Failed";
  }

  /**
   * Get the {@link PersonService} from the {@link CoinGuiceModule}
   * 
   * @return {@link PersonService}
   */
  protected PersonService getPersonService() {
    return CoinGuiceModule.getContext().getBean("personService",
        PersonService.class);
  }

  /**
   * Get the GroupService from the {@link CoinGuiceModule}
   * 
   * @return the {@link GroupService}
   */
  protected GroupService getGroupService() {
    return CoinGuiceModule.getContext().getBean("groupService",
        GroupService.class);
  }

  /**
   * @return the {@link ConextEnvironment}
   */
  protected ContextEnvironment getEnvironment() {
    return CoinGuiceModule.getContext().getBean("environment", ContextEnvironment.class);
  }

  /**
   * Execute a Http GET
   * 
   * @param url
   *          the url to call
   * @return the inputstream
   * @throws ClientProtocolException
   *           in case of Htpp protocol breach
   * @throws IOException
   *           in case of IO error
   */
  protected InputStream executeHttpGet(String url)
      throws ClientProtocolException, IOException {
    return httpClient.execute(new HttpGet(url)).getEntity().getContent();
  }

}