/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.lang.StringUtils;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpGet;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.social.core.oauth.OAuthSecurityToken;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.VirtualOrganizationCollectionOptions;
import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;

import nl.surfnet.coin.shared.log.ApiCallLog;
import nl.surfnet.coin.shared.log.ApiCallLogContextListener;
import nl.surfnet.coin.shared.log.ApiCallLogService;
import nl.surfnet.coin.shindig.oauth.CoinVoFilter;
import nl.surfnet.coin.shindig.protocol.ContextEnvironment;
import nl.surfnet.coin.shindig.protocol.HttpClientProvider;

/**
 * Base class for classes that delegate the functionality to a REST endpoint
 * (e.g. PHP EngineBlock)
 * 
 */
public abstract class AbstractRestDelegationService {
  /*
   * For executing http calls
   */
  @Autowired
  private HttpClientProvider httpClientProvider;

  /*
   * Context information
   */
  @Autowired
  private ContextEnvironment environment;
  
  @Autowired
  private ApiCallLogService logService;
  /*
   * For parsing the JSON
   */
  private static final ObjectMapper objectMapper = new ObjectMapper();

  /*
   * encoding format
   */
  protected static final String UTF_8 = "UTF-8";

  /**
   * @return the objectMapper
   */
  protected ObjectMapper getObjectMapper() {
    return objectMapper;
  }

  /**
   * Execute a Http GET
   * 
   * @param url
   *          the url to call
   * @return the inputstream
   * @throws ClientProtocolException
   *           in case of Htpp protocol breach
   * @throws IOException
   *           in case of IO error
   */
  protected InputStream executeHttpGet(String url)
      throws ClientProtocolException, IOException {
    return httpClientProvider.getHttpClient().execute(new HttpGet(url))
        .getEntity().getContent();
  }

  /**
   * @param httpClientProvider
   *          the httpClientProvider to set
   */
  public void setHttpClientProvider(HttpClientProvider httpClientProvider) {
    this.httpClientProvider = httpClientProvider;
  }

  /**
   * @return the environment
   */
  protected ContextEnvironment getEnvironment() {
    return environment;
  }

  /**
   * @param environment
   *          the environment to set
   */
  public void setEnvironment(ContextEnvironment environment) {
    this.environment = environment;
  }

  protected String getVirtualOrganizationContext(SecurityToken token,
      CollectionOptions options) {
    /**
     * There are three ways to detect a VirtualOrganization call. The
     * VirtualOrganizationCollectionOptions is for two-legged rest calls, the
     * ThreadLocal hack is for RPC two-legged calls and the SecurityToken check
     * for three-legged oauth
     */
    String voRpc = CoinVoFilter.getVirtualOrganizationContext();
    String virtualOrganization = null;
    if (options != null && options instanceof VirtualOrganizationCollectionOptions) {
      VirtualOrganizationCollectionOptions voOptions = (VirtualOrganizationCollectionOptions) options;
      virtualOrganization = voOptions.getVoName();
    } else if (!StringUtils.isBlank(voRpc)) {
      virtualOrganization = voRpc;
    } else if (token instanceof OAuthSecurityToken) {
      OAuthSecurityToken securityToken = (OAuthSecurityToken) token;
      if (!StringUtils.isBlank(securityToken.getVirtualOrganization())) {
        virtualOrganization = securityToken.getVirtualOrganization();
      }
    }
    return virtualOrganization;
  }
  
  protected String getServiceProviderContext(SecurityToken token) {
    String serviceProvider = null;
    if (token instanceof OAuthSecurityToken) {
      serviceProvider = ((OAuthSecurityToken) token).getServiceProviderEntityId();
    }
    return serviceProvider;
  }
  
  /*
   * Save the fact that a request is made
   */
  protected void logApiCall(SecurityToken token) {
    ApiCallLog log = ApiCallLogContextListener.getApiCallLog();
    log.setApiVersion("SHINDIG");
    String serviceProviderEntityId;
    if (token instanceof OAuthSecurityToken) {
      OAuthSecurityToken st = (OAuthSecurityToken) token;
      serviceProviderEntityId = st.getServiceProviderEntityId();
      log.setSpEntityId(serviceProviderEntityId);
    } else {
      serviceProviderEntityId = "NO_OAUTH";
    }
    String ownerId = token.getOwnerId();
    if (StringUtils.isBlank(ownerId)) {
      ownerId = serviceProviderEntityId;
    }
    log.setUserId(ownerId);
    log.setConsumerKey(token.getAppId());
    logService.saveApiCallLog(log);
    
  }

  /**
   * @param logService the logService to set
   */
  public void setLogService(ApiCallLogService logService) {
    this.logService = logService;
  }

}
