/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import nl.surfnet.coin.shared.domain.ErrorMail;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import nl.surfnet.coin.shindig.protocol.ContextEnvironment;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.util.ImmediateFuture;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.social.core.model.ActivityImpl;
import org.apache.shindig.social.opensocial.model.Activity;
import org.apache.shindig.social.opensocial.model.Group;
import org.apache.shindig.social.opensocial.model.ListField;
import org.apache.shindig.social.opensocial.model.Person;
import org.apache.shindig.social.opensocial.spi.ActivityService;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.PersonService;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Future;

/**
 * Implementation of {@link ActivityService} based on hibernate
 */
@Component(value = "activityService")
@Transactional
public class ActivityServiceImpl implements ActivityService,
        CoinActivityService {

  @Autowired
  private SessionFactory sessionFactory;

  @Autowired
  private OpenSocialValidator openSocialValidator;

  @Autowired
  private PersonService personService;

  @Autowired
  private ErrorMessageMailer errorMessageMailer;

  /*
   * Context information
   */
  @Autowired
  private ContextEnvironment environment;

  private static final String ERROR_MAILER_TEMPLATE = "exception_shindig_plain.txt";

  /*
   * Hibernate class
   */
  private static final String ACTIVITY = ActivityImpl.class.getName();

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.ActivityService#createActivity
   * (org.apache.shindig .social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.spi.GroupId, java.lang.String,
   * java.util.Set, org.apache.shindig.social.opensocial.model.Activity,
   * org.apache.shindig.auth.SecurityToken)
   */
  @Transactional
  public Future<Void> createActivity(UserId userId, GroupId groupId,
                                     String appId, Set<String> fields, Activity activity, SecurityToken token)
          throws ProtocolException {
    openSocialValidator.invariant(userId, groupId);
    if (!StringUtils.hasText(activity.getAppId())) {
      activity.setAppId(appId);
    }
    activity.setUserId(getPersonEmail(userId, token));
    activity.setPostedTime(System.currentTimeMillis());
    activity.setUpdated(new Date());
    if (groupId != null && StringUtils.hasText(groupId.getGroupId())) {
      activity.setExternalId(groupId.getGroupId());
    }
    sessionFactory.getCurrentSession().saveOrUpdate(activity);
    return null;
  }

  private String getPersonEmail(UserId userId, SecurityToken token)
          throws ProtocolException {
    try {
      Person person = personService.getPerson(userId, null, token).get();
      List<ListField> emails = person.getEmails();
      if (CollectionUtils.isEmpty(emails)) {
        return person.getDisplayName();
      } else {
        return emails.get(0).getValue();
      }
    } catch (Exception e) {
      String shortMessage = "Could not retrieve the Person display namefor userId('"
              + userId + "')";
      String errorMessage = "An exception occured in the Shindig server." +
              getClass().getSimpleName() + "<br/>" + shortMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage,
              errorMessage,
              e.getMessage(),
              getHost(),
              "Shindig");
      errorMail.setLocation(this.getClass().getName() + "#getPersonEmail");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new RuntimeException(shortMessage, e);
    }
  }

  private String getHost() {
    try {
      return InetAddress.getLocalHost().toString();
    } catch (UnknownHostException e) {
      return "UNKNOWN";
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.ActivityService#deleteActivities
   * (org.apache.shindig .social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.spi.GroupId, java.lang.String,
   * java.util.Set, org.apache.shindig.auth.SecurityToken)
   */
  @Transactional
  public Future<Void> deleteActivities(UserId userId, GroupId groupId,
                                       String appId, Set<String> activityIds, SecurityToken token)
          throws ProtocolException {
    openSocialValidator.invariant(userId, groupId);
    StringBuilder queryString = new StringBuilder("delete from " + ACTIVITY
            + " where appId = :appId ");
    if (groupId != null) {
      queryString.append(" and externalId = :externalId ");
    }
    if (userId != null) {
      queryString.append(" and userId = :userId");
    }
    Query query = sessionFactory.getCurrentSession().createQuery(
            queryString.toString());
    query.setString("appId", appId);
    if (groupId != null) {
      query.setString("groupId", groupId.getGroupId());
    }
    if (userId != null) {
      query.setString("userId", userId.getUserId(token));
    }
    query.executeUpdate();
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.ActivityService#getActivities(
   * java.util.Set, org.apache.shindig.social.opensocial.spi.GroupId,
   * java.lang.String, java.util.Set,
   * org.apache.shindig.social.opensocial.spi.CollectionOptions,
   * org.apache.shindig.auth.SecurityToken)
   */
  @SuppressWarnings("unchecked")
  @Transactional
  public Future<RestfulCollection<Activity>> getActivities(Set<UserId> userIds,
                                                           GroupId groupId, String appId, Set<String> fields,
                                                           CollectionOptions options, SecurityToken token) throws ProtocolException {
    openSocialValidator.invariant(userIds, groupId, appId);
    StringBuilder queryString = new StringBuilder("from " + ACTIVITY
            + " where 1 = 1 ");
    Collection<String> ids = null;
    if (StringUtils.hasText(appId)) {
      queryString.append("and appId = :appId ");
    }
    if (!CollectionUtils.isEmpty(userIds)) {
      ids = new ArrayList<String>();
      for (UserId userId : userIds) {
        ids.add(userId.getUserId(token));
        queryString.append(" and userId in (:ids) ");
      }
    }

    if (groupId != null && StringUtils.hasText(groupId.getGroupId())) {
      queryString.append(" and externalId = :externalId ");
    }
    Query query = sessionFactory.getCurrentSession().createQuery(
            queryString.toString());
    if (StringUtils.hasText(appId)) {
      query.setString("appId", appId);
    }
    if (!CollectionUtils.isEmpty(userIds)) {
      query.setParameterList("ids", ids);
    }
    if (groupId != null && StringUtils.hasText(groupId.getGroupId())) {
      query.setParameter("externalId", groupId.getGroupId());
    }
    List result = query.list();
    return ImmediateFuture.newInstance(new RestfulCollection<Activity>(result));
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.ActivityService#getActivities(
   * org.apache.shindig.social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.spi.GroupId, java.lang.String,
   * java.util.Set, org.apache.shindig.social.opensocial.spi.CollectionOptions,
   * java.util.Set, org.apache.shindig.auth.SecurityToken)
   */
  @Transactional
  public Future<RestfulCollection<Activity>> getActivities(UserId userId,
                                                           GroupId groupId, String appId, Set<String> fields,
                                                           CollectionOptions options, Set<String> activityIds, SecurityToken token)
          throws ProtocolException {
    if (activityIds != null && !activityIds.isEmpty()) {

    }
    return getActivities(
            new HashSet<UserId>(Collections.singletonList(userId)), groupId, appId,
            fields, options, token);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.ActivityService#getActivity(org
   * .apache.shindig.social .opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.spi.GroupId, java.lang.String,
   * java.util.Set, java.lang.String, org.apache.shindig.auth.SecurityToken)
   */
  @Transactional
  public Future<Activity> getActivity(UserId userId, GroupId groupId,
                                      String appId, Set<String> fields, String activityId, SecurityToken token)
          throws ProtocolException {
    Activity activity = (Activity) sessionFactory.getCurrentSession().load(
            ActivityImpl.class, activityId);
    return ImmediateFuture.newInstance(activity);
  }

  /**
   * @param sessionFactory the sessionFactory to set
   */
  public void setSessionFactory(SessionFactory sessionFactory) {
    this.sessionFactory = sessionFactory;
  }

  /**
   * @param openSocialValidator the openSocialId to set
   */
  public void setOpenSocialId(OpenSocialValidator openSocialValidator) {
    this.openSocialValidator = openSocialValidator;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.shindig.spi.CoinActivityService#findActivityByGroups(java
   * .util.Collection, int, int)
   */
  @SuppressWarnings("unchecked")
  @Override
  public List<Activity> findActivityByGroups(Group group, int maxResult,
                                             int firstResult) {
    Criteria criteria = sessionFactory.getCurrentSession().createCriteria(
            ActivityImpl.class);
    criteria.add(Restrictions.eq("externalId", group.getId().getGroupId()));
    criteria.setMaxResults(maxResult);
    criteria.setFirstResult(firstResult);
    criteria.addOrder(Order.desc("updated"));
    return criteria.list();
  }

  /**
   * @param personService the personService to set
   */
  public void setPersonService(PersonService personService) {
    this.personService = personService;
  }

  public void setErrorMessageMailer(ErrorMessageMailer errorMessageMailer) {
    this.errorMessageMailer = errorMessageMailer;
  }

  /**
   * @param environment the environment to set
   */
  public void setEnvironment(ContextEnvironment environment) {
    this.environment = environment;
  }

}
