/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Future;

import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.util.ImmediateFuture;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.social.core.model.MessageCollectionImpl;
import org.apache.shindig.social.core.model.MessageImpl;
import org.apache.shindig.social.opensocial.model.Message;
import org.apache.shindig.social.opensocial.model.MessageCollection;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.MessageService;
import org.apache.shindig.social.opensocial.spi.PersonService;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.hibernate.Query;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import nl.surfnet.coin.shindig.protocol.ContextEnvironment;

/**
 * Implementation of the OpenSocial {@link MessageService} spec
 * 
 * @author oharsta
 * 
 */
@Component(value = "messageService")
@Transactional
public class MessageServiceImpl implements MessageService {

  @Autowired
  private SessionFactory sessionFactory;

  /*
   * Context information
   */
  @Autowired
  private ContextEnvironment environment;
  
  /*
   * Hibernate class
   */
  private static final String MESSAGE = MessageImpl.class.getName();
  private static final String MESSAGE_COLLECTION = MessageCollectionImpl.class
      .getName();


  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#createMessage(org
   * .apache.shindig.social.opensocial.spi.UserId, java.lang.String,
   * java.lang.String, org.apache.shindig.social.opensocial.model.Message,
   * org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<Void> createMessage(UserId userId, String appId,
      String msgCollId, Message message, SecurityToken token)
      throws ProtocolException {
    MessageCollection messageCollection;
    if (StringUtils.hasText(msgCollId)) {
      messageCollection = (MessageCollection) sessionFactory
          .getCurrentSession().get(MESSAGE_COLLECTION, msgCollId);
      if (messageCollection == null) {
        throw new IllegalArgumentException("MessageCollection(ID='" + msgCollId
            + "') does not exist.");
      }
      List<String> collectionIds = message.getCollectionIds();
      if (collectionIds == null) {
        message.setCollectionIds(new ArrayList<String>());
        collectionIds = message.getCollectionIds();
      }
      if (!collectionIds.contains(msgCollId)) {
        collectionIds.add(msgCollId);
      }
    }
    if (message.getTimeSent() == null) {
      message.setTimeSent(new Date());
    }
    sessionFactory.getCurrentSession().saveOrUpdate(message);
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#createMessageCollection
   * (org.apache.shindig.social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.model.MessageCollection,
   * org.apache.shindig.auth.SecurityToken)
   */
  @SuppressWarnings("unchecked")
  @Override
  public Future<MessageCollection> createMessageCollection(UserId userId,
      MessageCollection msgCollection, SecurityToken token)
      throws ProtocolException {
    // first check if the msgCollection already exists
    if (!StringUtils.hasText(msgCollection.getTitle())) {
      throw new IllegalArgumentException(
          "Title is mandetory field for MessageCollection");
    }
    MessageCollection singleResult = (MessageCollection) DataAccessUtils
        .singleResult(sessionFactory.getCurrentSession().createQuery(
            "from " + MESSAGE_COLLECTION + " where title = :title").setString(
            "title", msgCollection.getTitle()).list());
    if (singleResult != null) {
      msgCollection = singleResult;
    } else {
      sessionFactory.getCurrentSession().saveOrUpdate(msgCollection);
    }
    return ImmediateFuture.newInstance(msgCollection);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#deleteMessageCollection
   * (org.apache.shindig.social.opensocial.spi.UserId, java.lang.String,
   * org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<Void> deleteMessageCollection(UserId userId, String msgCollId,
      SecurityToken token) throws ProtocolException {
    Query query = sessionFactory.getCurrentSession().createQuery(
        "delete from " + MESSAGE_COLLECTION + " where id = :id");
    query.setString("id", msgCollId);
    query.executeUpdate();
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#deleteMessages(
   * org.apache.shindig.social.opensocial.spi.UserId, java.lang.String,
   * java.util.List, org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<Void> deleteMessages(UserId userId, String msgCollId,
      List<String> ids, SecurityToken token) throws ProtocolException {
    deleteMessageCollection(userId, msgCollId, token);
    Query query = sessionFactory.getCurrentSession().createQuery(
        "delete from " + MESSAGE + " where id in (:id)");
    query.setParameterList("ids", ids);
    query.executeUpdate();
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#getMessageCollections
   * (org.apache.shindig.social.opensocial.spi.UserId, java.util.Set,
   * org.apache.shindig.social.opensocial.spi.CollectionOptions,
   * org.apache.shindig.auth.SecurityToken)
   */
  @SuppressWarnings("unchecked")
  @Override
  public Future<RestfulCollection<MessageCollection>> getMessageCollections(
      UserId userId, Set<String> fields, CollectionOptions options,
      SecurityToken token) throws ProtocolException {
    List result = sessionFactory.getCurrentSession().createQuery(
        "from " + MESSAGE_COLLECTION).list();
    return ImmediateFuture
        .newInstance(new RestfulCollection<MessageCollection>(result));
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#getMessages(org
   * .apache.shindig.social.opensocial.spi.UserId, java.lang.String,
   * java.util.Set, java.util.List,
   * org.apache.shindig.social.opensocial.spi.CollectionOptions,
   * org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<RestfulCollection<Message>> getMessages(UserId userId,
      String msgCollId, Set<String> fields, List<String> msgIds,
      CollectionOptions options, SecurityToken token) throws ProtocolException {
    
    String sortBy = options.getSortBy();
    if(sortBy.equals(PersonService.TOP_FRIENDS_SORT)) {
      // We don't support TOP_FRIENDS_SORT so fall back to timeSent by default
      sortBy = "timeSent";
    }
    sortBy = sortBy.replaceAll("[^a-zA-Z]", "");
    String sortOrder = options.getSortOrder().toString().equals("ascending") ? "asc" : "desc";
    
    /*
     * If messageIds are specified return them, otherwise return entries in the
     * given collection.
     */
    Query query;
    if (StringUtils.hasText(msgCollId)) {
      query = sessionFactory.getCurrentSession().createQuery(
          "from " + MESSAGE
              + " as mes where :msgCollId in elements(mes.collectionIds) order by "+sortBy+" "+sortOrder)
          .setString("msgCollId", msgCollId);
    } else {
      query = sessionFactory.getCurrentSession().createQuery(
          "from " + MESSAGE + " where id in (:msgIds)").setParameterList(
          "msgIds", msgIds);
    }
    query.setFirstResult(options.getFirst());
    query.setMaxResults(options.getMax());
    
    return ImmediateFuture.newInstance(new RestfulCollection<Message>(query.list()));
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#modifyMessage(org
   * .apache.shindig.social.opensocial.spi.UserId, java.lang.String,
   * java.lang.String, org.apache.shindig.social.opensocial.model.Message,
   * org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<Void> modifyMessage(UserId userId, String msgCollId,
      String messageId, Message message, SecurityToken token)
      throws ProtocolException {
    message.setId(messageId);
    sessionFactory.getCurrentSession().saveOrUpdate(message);
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.MessageService#modifyMessageCollection
   * (org.apache.shindig.social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.model.MessageCollection,
   * org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<Void> modifyMessageCollection(UserId userId,
      MessageCollection msgCollection, SecurityToken token)
      throws ProtocolException {
    sessionFactory.getCurrentSession().saveOrUpdate(msgCollection);
    return null;
  }

  /**
   * @param environment the environment to set
   */
  public void setEnvironment(ContextEnvironment environment) {
    this.environment = environment;
  }
  
}
