/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import java.util.Set;

import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

/**
 * Responsible for validating Id's for OpenSocial queries.
 * 
 */
@Component(value = "openSocialId")
public class OpenSocialValidator {

  /**
   * 
   * Check if there are userIds to identify
   * 
   * @param userIds
   *          all userIds
   * @param groupId
   *          the groupId
   */
  public void invariant(Set<UserId> userIds, GroupId groupId) {
    if (groupId == null && (userIds == null || userIds.isEmpty())) {
      throw new IllegalArgumentException(
          "Either groupId or userIds must be provided");
    }
    invariant(groupId);
  }

  /**
   * 
   * Check if there are userIds to identify
   * 
   * @param userIds
   *          all userIds
   * @param groupId
   *          the groupId
   * @param appId
   *          the application Id
   */
  public void invariant(Set<UserId> userIds, GroupId groupId, String appId) {
    if (groupId == null && (userIds == null || userIds.isEmpty())
        && !StringUtils.hasText(appId)) {
      throw new IllegalArgumentException(
          "Either groupId, userIds or appId must be provided");
    }
    invariant(groupId);
  }

  /**
   * 
   * Check if there are userIds to identify
   * 
   * @param userId
   *          the userId
   * @param groupId
   *          the groupId
   */
  public void invariant(UserId userId, GroupId groupId) {
    if (userId == null && groupId == null) {
      throw new IllegalArgumentException(
          "Either groupId or userId must be provided");
    }
    invariant(groupId);
  }

  /**
   * We only support {@link GroupId.Type.groupId}
   * 
   * @param groupId
   *          the groupId
   */
  public void invariant(GroupId groupId) {
    /*
     * if (groupId != null && !groupId.getType().equals(GroupId.Type.groupId)) {
     * 
     * We don't support anything but a GroupId.groupId
     * 
     * throw new IllegalArgumentException("We don't support the GroupType '" +
     * groupId + "'"); }
     */
  }

  /**
   * Check if the userId is valid
   * 
   * @param userId
   *          the userId
   */
  public void invariant(UserId userId) {
    if (userId == null) {
      throw new IllegalArgumentException("userId must be provided");
    }
  }

  /**
   * Check if the userIds are valid
   * 
   * @param userIds
   *          the userIds
   */
  public void invariant(Set<UserId> userIds) {
    invariant(userIds, false);
  }

  /**
   * Check if the userIds are valid
   * 
   * @param userIds
   *          the userIds
   */
  public void invariant(Set<UserId> userIds, boolean requiresSingleUser) {
    if (userIds == null || userIds.isEmpty()) {
      throw new IllegalArgumentException("At least one userId must be provided");
    }
    if (requiresSingleUser && userIds.size() > 1) {
      throw new IllegalArgumentException(
          "Only one userId may be specified. Multiple userIds are not supported");
    }
  }
}
