/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.commons.io.IOUtils;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.util.ImmediateFuture;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.social.core.model.AccountImpl;
import org.apache.shindig.social.core.model.ListFieldImpl;
import org.apache.shindig.social.core.model.NameImpl;
import org.apache.shindig.social.core.model.OrganizationImpl;
import org.apache.shindig.social.core.model.PersonImpl;
import org.apache.shindig.social.core.oauth.OAuthSecurityToken;
import org.apache.shindig.social.opensocial.model.Account;
import org.apache.shindig.social.opensocial.model.Group;
import org.apache.shindig.social.opensocial.model.ListField;
import org.apache.shindig.social.opensocial.model.Organization;
import org.apache.shindig.social.opensocial.model.Person;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.GroupService;
import org.apache.shindig.social.opensocial.spi.PersonService;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.codehaus.jackson.map.JsonMappingException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import nl.surfnet.coin.shared.domain.ErrorMail;
import nl.surfnet.coin.shared.log.ApiCallLog;
import nl.surfnet.coin.shared.log.ApiCallLogContextListener;
import nl.surfnet.coin.shared.log.ApiCallLogService;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;

/**
 * Implementation of shindig {@link PersonService} connecting to the EngineBlock
 * PHP endpoint
 */
@Component(value = "personService")
public class PersonServiceImpl extends AbstractRestDelegationService implements
    PersonService {

  /*
   * Extension on the baseSocialApiUrl for the people functionality
   */
  private static final String PERSON_URL = "/social/people/";

  @Autowired
  private ErrorMessageMailer errorMessageMailer;

  @Autowired
  private GroupService groupService;

  @Autowired
  private OpenSocialValidator openSocialValidator;
  

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.PersonService#getPeople(java.util
   * .Set, org.apache.shindig.social.opensocial.spi.GroupId,
   * org.apache.shindig.social.opensocial.spi.CollectionOptions, java.util.Set,
   * org.apache.shindig.auth.SecurityToken)
   */
  public Future<RestfulCollection<Person>> getPeople(Set<UserId> userIds,
      GroupId groupId, CollectionOptions collectionOptions, Set<String> fields,
      SecurityToken token) throws ProtocolException {
    openSocialValidator.invariant(userIds, groupId);
    String onBehalfOf = (token != null ? token.getOwnerId() : null);
    try {
      this.logApiCall(token);
      return doGetPeople(userIds, groupId, fields, token, collectionOptions,
          onBehalfOf);
    } catch (Exception e) {
      String shortMessage = "Exception in getPeople for user(id='" + onBehalfOf
          + "')";
      String errorMessage = "An exception occured in the PersonService in the Shindig server<br/>"
          + "connecting to the EngineBlock PHP endpoint:"
          + e.getMessage()
          + "<br/><br/>" + getClass().getSimpleName() + "<br/>" + shortMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage, errorMessage,
          onBehalfOf, e.getMessage(), getHost(), "Shindig");
      errorMail.setLocation(this.getClass().getName() + "#getPeople");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new RuntimeException(errorMessage, e);
    }
  }

  private String getHost() {
    try {
      return InetAddress.getLocalHost().toString();
    } catch (UnknownHostException e) {
      return "UNKNOWN";
    }
  }

  /*
   * doGetPeople
   */
  private Future<RestfulCollection<Person>> doGetPeople(Set<UserId> userIds,
      GroupId groupId, Set<String> fields, SecurityToken token,
      CollectionOptions collectionOptions, String onBehalfOf)
      throws IOException, InterruptedException, ExecutionException {
    List<Person> persons = new ArrayList<Person>();
    if (groupId == null) {
      if (userIds.size() > 1) {
        /*
         * we are dealing with multiple persons, but no groupId
         */
        String idsOfUser = org.apache.commons.lang.StringUtils.join(userIds,
            ",");
        persons = doGetPersons(idsOfUser, null, token, collectionOptions,
            onBehalfOf);
      } else {
        /*
         * Single person to return
         */
        Future<Person> person = getPerson(userIds.iterator().next(), fields,
            token);
        persons = Collections.singletonList(person.get());
      }
    } else {
      persons = doGetPersons(userIds.iterator().next().getUserId(token),
          groupId.getGroupId(), token, collectionOptions, onBehalfOf);
    }
    return createFutureForPersons(persons);
  }

  private Future<RestfulCollection<Person>> createFutureForPersons(
      List<Person> persons) {
    if (persons == null) {
      persons = new ArrayList<Person>();
    }
    return ImmediateFuture.newInstance(new RestfulCollection<Person>(persons));
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.PersonService#getPerson(org.apache
   * .shindig.social .opensocial.spi.UserId, java.util.Set,
   * org.apache.shindig.auth.SecurityToken)
   */
  public Future<Person> getPerson(UserId id, Set<String> fields,
      SecurityToken token) throws ProtocolException {
    try {
      String idOfUser = id.getUserId(token);
      String onBehalfOf = token != null ? token.getOwnerId() : null;
      List<Person> persons = doGetPersons(idOfUser, null, token, null,
          onBehalfOf);
      Person person;
      if (persons == null || persons.size() == 0) {
        person = null;
      } else {
        person = persons.get(0);
      }
      this.logApiCall(token);
      return ImmediateFuture.newInstance(person);
    } catch (Exception e) {
      String shortMessage = "Exception in getPerson(id='" + id.getUserId(token)
          + "')";
      String errorMessage = "An exception occured in the PersonService in the Shindig server\n"
          + "connecting to the EngineBlock PHP endpoint:<br/><br/>"
          + getClass().getSimpleName() + "<br/>" + shortMessage;
      ErrorMail errorMailDetails = new ErrorMail(shortMessage, errorMessage,
          id.getUserId(token), e.getMessage(), getHost(), "Shindig");

      errorMessageMailer.sendErrorMail(errorMailDetails);
      throw new RuntimeException(shortMessage, e);
    }
  }

  private List<Person> doGetPersons(String formattedIds, String groupId,
      SecurityToken securityToken, CollectionOptions collectionOptions,
      String onBehalfOf) throws IOException, InterruptedException,
      ExecutionException {
    StringBuilder url = new StringBuilder(getEnvironment()
        .getBaseSocialApiUrl());
    url.append(PERSON_URL);
    boolean subjectIsNotOwner = StringUtils.hasText(onBehalfOf)
        && !formattedIds.equals(onBehalfOf);
    if (subjectIsNotOwner) {
      // check if the onBehalfOf is a member of the same group as the
      // formattedIds
      checkIdentitySecurity(onBehalfOf, formattedIds, securityToken);
      url.append(URLEncoder.encode(onBehalfOf, UTF_8));
    } else {
      url.append(URLEncoder.encode(formattedIds, UTF_8));
    }
    if (StringUtils.hasText(groupId)) {
      url.append("/");
      url.append(URLEncoder.encode(groupId, UTF_8));
    }
    url.append("?fields=all");
    String serviceProviderContext = super
        .getServiceProviderContext(securityToken);
    if (StringUtils.hasText(serviceProviderContext)) {
      url.append("&sp-entity-id="
          + URLEncoder.encode(serviceProviderContext, UTF_8));
    }
    String virtualOrganizationContext = super.getVirtualOrganizationContext(
        securityToken, collectionOptions);
    if (StringUtils.hasText(virtualOrganizationContext)) {
      url.append("&vo=" + URLEncoder.encode(virtualOrganizationContext, UTF_8));
    }
    if (subjectIsNotOwner) {
      url.append("&subject-id=" + URLEncoder.encode(formattedIds, UTF_8));
    }
    InputStream inputStream = executeHttpGet(url.toString());
    List<Person> persons = constructPersons(inputStream,
        virtualOrganizationContext);
    inputStream.close();
    /*
     * If the query was for team members (e.g. groupId != null) then the
     * onBehalf must be a member of that Group. We check after the query has
     * been done for performance optimization
     */
    if (StringUtils.hasText(groupId) && StringUtils.hasText(onBehalfOf)) {
      checkIdentitySecurityGroup(groupId, onBehalfOf, persons);
    }
    return persons;
  }

  private void checkIdentitySecurityGroup(String groupId, String onBehalfOf,
      List<Person> persons) {
    if (persons != null) {
      for (Person person : persons) {
        if (person.getId().equals(onBehalfOf)) {
          return;
        }
      }
    }
    throw new IllegalArgumentException("Requested members retrieval for "
        + groupId + " is not granted, as " + onBehalfOf + " is not a member");
  }

  private void checkIdentitySecurity(String onBehalfOf, String personId,
      SecurityToken token) throws InterruptedException, ExecutionException,
      IOException {
    /*
     * We get the groups of the personId and check if the onBehalfOf is a part
     * of any of those groups. If this is not the case then we throw an
     * Exception
     */

    List<Group> personIdGroups = this.groupService
        .getGroups(new UserId(UserId.Type.userId, personId), null, null,
            getSecureToken(personId, token)).get().getEntry();
    List<Group> onBehalfOfGroups = this.groupService
        .getGroups(new UserId(UserId.Type.userId, onBehalfOf), null, null,
            getSecureToken(onBehalfOf, token)).get().getEntry();
    Set<Group> merged = new HashSet<Group>(personIdGroups);
    merged.addAll(onBehalfOfGroups);

    int personGroupsSize = personIdGroups.size();
    int onBehalfOfGroupsSize = onBehalfOfGroups.size();
    int mergedSize = merged.size();
    if (personGroupsSize + onBehalfOfGroupsSize == mergedSize) {
      throw new IllegalArgumentException("Requested person retrieval for "
          + personId + " is not granted, as " + onBehalfOf
          + " does not share a teammembership");
    }
  }

  /*
   * In order to assess if a person can retrieve information about someone, they
   * must share groups. To retrieve the groups however we need a token where the
   * owner == user
   */
  private SecurityToken getSecureToken(String personId, SecurityToken token) {
    if (token instanceof OAuthSecurityToken) {
      OAuthSecurityToken oauthToken = (OAuthSecurityToken) token;
      return new OAuthSecurityToken(personId, token.getAppUrl(),
          token.getAppId(), token.getDomain(), token.getDomain(),
          token.getExpiresAt(), oauthToken.getVirtualOrganization(),
          oauthToken.getServiceProviderEntityId());

    }
    return new OAuthSecurityToken(personId, token.getAppUrl(),
        token.getAppId(), token.getDomain(), token.getDomain(),
        token.getExpiresAt(), null, null);
  }

  @SuppressWarnings({ "unchecked", "rawtypes" })
  private List<Person> constructPersons(InputStream inputStream,
      String virtualOrganizationContext) throws IOException {
    String json = null;
    try {
      json = IOUtils.toString(inputStream);
      HashMap<String, Object> result = getObjectMapper().readValue(json,
          HashMap.class);
      List entries = (List) result.get("entry");
      if (CollectionUtils.isEmpty(entries) || isEmptyListOfList(entries)) {
        return null;
      }
      List<Person> persons = new ArrayList<Person>();
      List<Map> entry;
      if (entries.get(0) instanceof List) {
        List l = (List) entries.get(0);
        entry = l;
      } else {
        entry = entries;
      }
      /*
       * { "entry":[ { "name":{ "formatted":"Hans Zandbelt", "givenName":"Hans",
       * "familyName":"Zandbelt" }, "id":"urn:collab:person:surfnet.nl:hansz",
       * "displayName":"Hans Zandbelt", "nickname":"Hans Zandbelt", "emails":[
       * "Hans.Zandbelt@surfnet.nl" ], "organizations":{ "name":"surfnet.nl" },
       * "account":{ "username":"hansz", "userId":"hansz" }, "tags":["member"] }
       * ], "startIndex":0, "totalResults":1, "itemsPerPage":20 }
       */
      for (Map personMap : entry) {
        Person person = new PersonImpl();
        person.setId((String) personMap.get("id"));
        person.setNickname((String) personMap.get("nickname"));
        person.setDisplayName((String) personMap.get("displayName"));
        person.setVoot_membership_role((String) personMap
            .get("voot_membership_role"));

        List<String> emailList = (List<String>) personMap.get("emails");
        if (!CollectionUtils.isEmpty(emailList)) {
          List<ListField> emails = new ArrayList<ListField>();
          for (String email : emailList) {
            emails.add(new ListFieldImpl("email", email));
          }
          person.setEmails(emails);
        }
        Object organizationMapWrapper = personMap.get("organizations");
        if (organizationMapWrapper instanceof Map
            && !CollectionUtils.isEmpty((Map) (organizationMapWrapper))) {
          Map organizationMap = (Map) organizationMapWrapper;
          List<Organization> organizations = new ArrayList<Organization>();
          OrganizationImpl organization = new OrganizationImpl();
          organization.setName((String) organizationMap.get("name"));
          organizations.add(organization);
          person.setOrganizations(organizations);
        }

        Object nameMapWrapper = personMap.get("name");
        if (nameMapWrapper instanceof Map
            && !CollectionUtils.isEmpty((Map) nameMapWrapper)) {
          Map nameMap = (Map) nameMapWrapper;
          NameImpl name = new NameImpl();
          name.setFormatted((String) nameMap.get("formatted"));
          name.setGivenName((String) nameMap.get("givenName"));
          name.setFamilyName((String) nameMap.get("familyName"));
          person.setName(name);
        }
        Object accountMapWrapper = personMap.get("account");
        if (accountMapWrapper instanceof Map
            && !CollectionUtils.isEmpty((Map) accountMapWrapper)) {
          Map accountMap = (Map) personMap.get("account");
          List<Account> accounts = new ArrayList<Account>();
          accounts.add(new AccountImpl((String) accountMap.get("domain"),
              (String) accountMap.get("userId"), (String) accountMap
                  .get("username")));
          person.setAccounts(accounts);
        }
        Object extensionsMapWrapper = personMap.get(virtualOrganizationContext);
        if (extensionsMapWrapper instanceof List
            && !CollectionUtils.isEmpty((Map) extensionsMapWrapper)) {
          List<Map> extensionsMap = (List) extensionsMapWrapper;
          Map<String, String> extensions = new HashMap<String, String>();
          for (Object object : extensionsMap) {
            Map map = (Map) object;
            Map keyValue = (Map) map.get("entry");
            String key = (String) keyValue.get("key");
            String value = "";
            for (String val : (ArrayList<String>) keyValue.get("value")) {
              value += val + ", ";
            }
            value = org.apache.commons.lang.StringUtils.removeEnd(value.trim(),
                ",");
            extensions.put(key, value);
          }
          person.setExtensions(extensions);
        }
        Object tagsListWrapper = personMap.get("tags");
        if (tagsListWrapper instanceof List
            && !CollectionUtils.isEmpty((List) tagsListWrapper)) {
          List<String> tagsList = (List<String>) tagsListWrapper;
          person.setTags(tagsList);

        }
        persons.add(person);
      }
      return persons;
    } catch (EOFException e) {
      // matter of contract
      return null;
    } catch (Exception e) {
      String errorMessage = "Error parsing Person JSON: " + json;
      throw new RuntimeException(errorMessage, e);
    }
  }

  private boolean isEmptyListOfList(List entries) {
    return (entries.get(0) instanceof List && ((List) entries.get(0)).size() == 0);
  }

  /**
   * @param openSocialValidator
   *          the openSocialId to set
   */
  public void setOpenSocialId(OpenSocialValidator openSocialValidator) {
    this.openSocialValidator = openSocialValidator;
  }

  public void setErrorMessageMailer(ErrorMessageMailer errorMessageMailer) {
    this.errorMessageMailer = errorMessageMailer;
  }

  /**
   * @param groupService
   *          the groupService to set
   */
  protected void setGroupService(GroupService groupService) {
    this.groupService = groupService;
  }
}
