/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import static org.junit.Assert.assertEquals;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.protocol.model.SortOrder;
import org.apache.shindig.social.core.model.MessageCollectionImpl;
import org.apache.shindig.social.core.model.MessageImpl;
import org.apache.shindig.social.opensocial.model.Message;
import org.apache.shindig.social.opensocial.model.Message.Status;
import org.apache.shindig.social.opensocial.model.MessageCollection;
import org.apache.shindig.social.opensocial.model.Url;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.MessageService;
import org.hibernate.SessionFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.transaction.TransactionConfiguration;
import org.springframework.transaction.annotation.Transactional;

import nl.surfnet.coin.shindig.model.PersistentUrl;

/**
 * {@link Test} for {@link MessageServiceImpl}
 * 
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = {
    "classpath:coin-shindig-hsqldb-properties-context.xml",
    "classpath:coin-shindig-context.xml",
    "classpath:coin-shared-context.xml" })
@TransactionConfiguration(transactionManager = "transactionManager", defaultRollback = true)
@Transactional
public class MessageServiceImplTest {

  @Autowired
  private MessageService messageService;

  @Autowired
  private SessionFactory sessionFactory;

  private String group1 = "group1";

  private String messageId;
  private String messageCollectionId;

  @Before
  public void setupDbContent() {
    MessageCollection msgColl = new MessageCollectionImpl();
    msgColl.setTitle(group1);
    Url url = new PersistentUrl();
    url.setLinkText("linkText");
    url.setValue("value");
    msgColl.setUrls(Collections.singletonList(url));
    messageService.createMessageCollection(null, msgColl, null);
    messageCollectionId = msgColl.getId();

    Message message = new MessageImpl();
    message.setStatus(Status.NEW);
    message.setBody("body");
    message.setCollectionIds(Collections.singletonList(msgColl.getId()));
    messageService.createMessage(null, null, null, message, null);
    messageId = message.getId();

    sessionFactory.getCurrentSession().flush();
  }

  @After
  public void flushSession() {
    // flushing forces the actual sql to be executed
    sessionFactory.getCurrentSession().flush();
  }

  @Test
  public void testGetMessage() throws InterruptedException, ExecutionException {
    CollectionOptions options = new CollectionOptions();
    options.setSortBy("timeSent");
    options.setSortOrder(SortOrder.ascending);
    options.setFirst(0);
    options.setMax(20);
    Future<RestfulCollection<Message>> result = messageService.getMessages(
        null, messageCollectionId, null, null, options, null);
    List<Message> messages = result.get().getEntry();
    assertEquals(1, messages.size());

    result = messageService.getMessages(null, null, null, Collections
        .singletonList(messageId), options, null);
    messages = result.get().getEntry();
    assertEquals(1, messages.size());
  }

  @Test
  public void testGetMessageCollection() throws InterruptedException,
      ExecutionException {
    Future<RestfulCollection<MessageCollection>> result = messageService
        .getMessageCollections(null, null, null, null);
    List<MessageCollection> messages = result.get().getEntry();
    assertEquals(1, messages.size());
  }

  @Test
  public void testCreateMessageCollection() throws InterruptedException,
      ExecutionException {
    MessageCollection mc = new MessageCollectionImpl();
    mc.setTitle(group1);
    mc = messageService.createMessageCollection(null, mc, null).get();
    assertEquals(messageCollectionId, mc.getId());

  }
}
