/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import nl.surfnet.coin.mock.AbstractMockHttpServerTest;
import nl.surfnet.coin.shared.log.ApiCallLog;
import nl.surfnet.coin.shared.log.ApiCallLogServiceImpl;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import nl.surfnet.coin.shindig.protocol.ContextEnvironment;
import nl.surfnet.coin.shindig.protocol.HttpClientProvider;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.social.core.oauth.OAuthSecurityToken;
import org.apache.shindig.social.opensocial.model.Organization;
import org.apache.shindig.social.opensocial.model.Person;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.apache.shindig.social.opensocial.spi.UserId.Type;
import org.junit.Assert;
import org.junit.Test;
import org.mockito.internal.stubbing.answers.Returns;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.jdbc.core.JdbcTemplate;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Future;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

/**
 * {@link Test} for {@link PersonServiceImpl}
 * 
 */
public class PersonServiceImplTest extends AbstractMockHttpServerTest {

  /*
   * Holds no state
   */
  private PersonServiceImpl personService;

  /**
   * Constructor
   */
  public PersonServiceImplTest() {
    super();
    personService = new PersonServiceImpl();
    ContextEnvironment environment = new ContextEnvironment();
    environment.setBaseSocialApiUrl("http://localhost:8088/social/person");
    personService.setEnvironment(environment);
    personService.setHttpClientProvider(new HttpClientProvider());
    personService.setOpenSocialId(new OpenSocialValidator());
    ApiCallLogServiceMock logService = new ApiCallLogServiceMock();
    personService.setLogService(logService);
    ErrorMessageMailer errorMessageMailer = new ErrorMessageMailer();
    personService.setErrorMessageMailer(errorMessageMailer);

    GroupServiceImpl groupService = new GroupServiceImpl();
    groupService.setEnvironment(environment);
    groupService.setErrorMessageMailer(errorMessageMailer);
    groupService.setHttpClientProvider(new HttpClientProvider());
    groupService.setLogService(logService);
    
    personService.setGroupService(groupService);
  }

  /**
   * Test method for
   * {@link nl.surfnet.coin.shindig.spi.PersonServiceImpl#getPerson(org.apache.shindig.social.opensocial.spi.UserId, java.util.Set, org.apache.shindig.auth.SecurityToken)}
   * .
   */
  @Test
  public void testGetPerson() throws Exception {
    UserId id = new UserId(Type.userId, "urn:collab:person:surfnet.nl:hansz");
    SecurityToken token = getMockToken(id);
    setResponseResource(new ClassPathResource("json/mock-single-person.json"));
    Person person = personService.getPerson(id, null, token).get();
    assertEquals("Hans Zandbelt", person.getDisplayName());
    assertEquals("Hans.Zandbelt@surfnet.nl", person.getEmails().get(0)
        .getValue());

    setResponseResource(new ClassPathResource("json/mock-empty-person.json"));
    person = personService.getPerson(id, null, token).get();
    assertNull(person);
  }

  /**
   * @param id
   * @return
   */
  private SecurityToken getMockToken(UserId id) {
    SecurityToken mock = new OAuthSecurityToken(id.getUserId(), "appUrl", "appId", "domain",
        "container", 0L, "virtualOrganization", "serviceProviderEntityId");
    return mock;
  }

  /**
   * Test method for
   * {@link PersonServiceImpl#getPeople(Set, org.apache.shindig.social.opensocial.spi.GroupId, org.apache.shindig.social.opensocial.spi.CollectionOptions, Set, SecurityToken)}
   * 
   * @throws Exception
   *           unexpected
   */
  @Test
  public void testGetPeople() throws Exception {
    UserId id = new UserId(Type.userId, "urn:collab:person:surfnet.nl:hansz");
    SecurityToken token = getMockToken(id);

    setResponseResource(new ClassPathResource("json/mock-multiple-persons.json"));
    HashSet<UserId> userIds = new HashSet<UserId>();
    userIds.add(id);
    Future<RestfulCollection<Person>> people = personService.getPeople(userIds,
        new GroupId(
            org.apache.shindig.social.opensocial.spi.GroupId.Type.groupId,
            "testGroup"), null, null, token);
    List<Person> persons = people.get().getEntry();
    assertEquals(2, persons.size());

  }

  /**
   * Test method for
   * {@link PersonServiceImpl#getPeople(Set, org.apache.shindig.social.opensocial.spi.GroupId, org.apache.shindig.social.opensocial.spi.CollectionOptions, Set, SecurityToken)}
   * 
   * @throws Exception
   *           unexpected
   */
  @Test
  public void testGetMultiplePeople() throws Exception {
    UserId id = new UserId(Type.userId, "urn:collab:person:surfnet.nl:hansz");
    SecurityToken token = getMockToken(id);
    setResponseResource(new ClassPathResource("json/mock-multiple-persons.json"));
    HashSet<UserId> userIds = new HashSet<UserId>();
    userIds.add(new UserId(Type.userId, "1"));
    userIds.add(new UserId(Type.userId, "2"));
    Future<RestfulCollection<Person>> people = personService.getPeople(userIds,
        null, null, null, token);

    List<Person> persons = people.get().getEntry();
    assertEquals(2, persons.size());

  }

  /**
   * Test method for
   * {@link PersonServiceImpl#getPeople(Set, org.apache.shindig.social.opensocial.spi.GroupId, org.apache.shindig.social.opensocial.spi.CollectionOptions, Set, SecurityToken)}
   * 
   * @throws Exception
   *           unexpected
   */
  @Test
  public void testGetMultiplePeopleFromGroup() throws Exception {
    UserId id = new UserId(Type.userId, "urn:collab:person:surfnet.nl:hansz");
    SecurityToken token = getMockToken(id);

    setResponseResource(new ClassPathResource("json/mock-tu-persons.json"));

    HashSet<UserId> userIds = new HashSet<UserId>();
    userIds.add(id);
    GroupId groupId = new GroupId(
        org.apache.shindig.social.opensocial.spi.GroupId.Type.groupId, "test");
    Future<RestfulCollection<Person>> people = personService.getPeople(userIds,
        groupId, null, null, token);

    List<Person> persons = people.get().getEntry();
    assertEquals(4, persons.size());

  }

  /**
   * Test method for
   * {@link PersonServiceImpl#getPeople(Set, org.apache.shindig.social.opensocial.spi.GroupId, org.apache.shindig.social.opensocial.spi.CollectionOptions, Set, SecurityToken)}
   * 
   * @throws Exception
   *           unexpected
   */
  @Test
  public void testGetPeopleFromArrayInArray() throws Exception {
    UserId id = new UserId(Type.userId, "urn:collab:person:surfnet.nl:hansz");
    SecurityToken token = getMockToken(id);

    setResponseResource(new ClassPathResource("json/array-in-array-hz.json"));

    HashSet<UserId> userIds = new HashSet<UserId>();
    userIds.add(id);
    Future<RestfulCollection<Person>> people = personService.getPeople(userIds,
        null, null, null, token);

    List<Person> persons = people.get().getEntry();
    assertEquals(1, persons.size());

  }

  @Test
  public void testGetOrganisationFromPerson() throws Exception {
    UserId id = new UserId(Type.userId, "whatever");
    SecurityToken token = getMockToken(id);
    setResponseResource(new ClassPathResource(
        "json/mock-single-person-with-organization.json"));
    Person person = personService.getPerson(id, null, token).get();
    List<Organization> organizations = person.getOrganizations();
    assertEquals(1, organizations.size());
    Organization organization = organizations.get(0);
    String name = organization.getName();
    assertEquals(name, "test.surfguest.nl");
  }

  /**
   * Test if we get a security exception when trying to get personInfo with an
   * onBehalfOf (three-legged oauth) that does not share a group
   * 
   * @throws Exception
   */
  @Test
  public void testSecurityCheck() throws Exception {
    UserId id = new UserId(Type.userId, "secret-person");
    SecurityToken token = new OAuthSecurityToken("onBehalfOf", "appUrl",
        "appId", "domain", "container", Long.MAX_VALUE, "authMode", /* virtualOrganization */
        null, /* serviceProviderEntityId */null);
    setResponseResource(new Resource[] {
        new ClassPathResource("json/mock-multiple-teams.json"),
        new ClassPathResource(
            "json/mock-multiple-teams-for-identity-check.json"),
        new ClassPathResource("json/mock-single-person-with-organization.json") });
    try {
      personService.getPerson(id, null, token).get();
      Assert.fail("IllegalArgumentException because of security breach ");
    } catch (RuntimeException e) {
      String message = e.getCause().getMessage();
      Assert.assertTrue(message.contains("not granted"));
    }
  }

  /**
   * Test if we get a security exception when trying to get personInfo with an
   * onBehalfOf (three-legged oauth) that does not share a group
   * 
   * @throws Exception
   */
  @Test
  public void testSecurityCheckHappyFlow() throws Exception {
    UserId id = new UserId(Type.userId, "secret-person");
    SecurityToken token = new OAuthSecurityToken("onBehalfOf", "appUrl",
        "appId", "domain", "container", Long.MAX_VALUE, "authMode", /* virtualOrganization */
        null, /* serviceProviderEntityId */null);
    setResponseResource(new Resource[] {
        new ClassPathResource("json/mock-multiple-teams.json"),
        new ClassPathResource("json/mock-multiple-teams.json"),
        new ClassPathResource("json/mock-single-person-with-organization.json") });

    personService.getPerson(id, null, token).get();

  }

  @Test
  public void testGetTeamMebersSecurityException() throws Exception {
    UserId id = new UserId(Type.userId, "urn:collab:person:surfnet.nl:hansz");
    SecurityToken token = new OAuthSecurityToken("onBehalfOf", "appUrl",
        "appId", "domain", "container", Long.MAX_VALUE, "authMode", /* virtualOrganization */
        null, /* serviceProviderEntityId */null);

    setResponseResource(new ClassPathResource("json/mock-multiple-persons.json"));
    HashSet<UserId> userIds = new HashSet<UserId>();
    userIds.add(id);
    try {
      personService.getPeople(userIds, new GroupId(
          org.apache.shindig.social.opensocial.spi.GroupId.Type.groupId,
          "testGroup"), null, null, token);
      Assert.fail("Expected security breach exception");
    } catch (RuntimeException e) {
      Assert.assertTrue(e.getCause().getMessage().contains("not granted"));
    }

  }
}
