/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.oauth;

import java.io.ByteArrayInputStream;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.HashMap;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.apache.shindig.auth.AuthenticationMode;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.crypto.Crypto;
import org.apache.shindig.social.core.oauth.OAuthSecurityToken;
import org.apache.shindig.social.opensocial.oauth.OAuthDataStore;
import org.apache.shindig.social.opensocial.oauth.OAuthEntry;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import net.oauth.OAuthConsumer;
import net.oauth.OAuthProblemException;
import net.oauth.OAuthServiceProvider;
import net.oauth.signature.RSA_SHA1;
import nl.surfnet.coin.shared.domain.ErrorMail;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import nl.surfnet.coin.shindig.spi.AbstractRestDelegationService;
import nl.surfnet.coin.shindig.spi.OAuthEntryService;

/**
 * {@link OAuthDataStore} that retrieves the values from EngineBlock
 * <p/>
 * Usage scenario: coin container is the oauth provider (oauth data are stored
 * here, e.g. iGoogle requests data from Coin)
 */
@Component(value = "oAuthDataStore")
public class CoinOAuthDataStore extends AbstractRestDelegationService implements
        OAuthDataStore {

  // This needs to be long enough that an attacker can't guess it. If the
  // attacker can guess this
  // value before they exceed the maximum number of attempts, they can complete
  // a session fixation
  // attack against a user.
  private static final int CALLBACK_TOKEN_LENGTH = 6;

  // We limit the number of trials before disabling the request token.
  private static final int CALLBACK_TOKEN_ATTEMPTS = 5;

  /**
   * The additional attributes that are added to a consumer upon fetching it
   * from the db (janus in our case).
   */
  private static final ImmutableList<String> ADDITIONAL_PROPERTIES = ImmutableList
          .of("coin:oauth:app_title", "coin:oauth:app_description",
                  "coin:oauth:app_thumbnail", "coin:oauth:app_icon",
                  OAuthSecurityToken.ENTITY_ID);

  private final OAuthServiceProvider SERVICE_PROVIDER;
  private static final String DOMAIN = "surfnet.nl";

  @Autowired
  private OAuthEntryService oAuthEntryService;

  @Autowired
  private ErrorMessageMailer errorMessageMailer;

  private Logger logger = Logger.getLogger(this.getClass());

  public CoinOAuthDataStore() {
    String baseUrl = "/oauth/";
    SERVICE_PROVIDER = new OAuthServiceProvider(baseUrl + "requestToken",
            baseUrl + "authorize", baseUrl + "accessToken");
  }

  @Override
  public OAuthEntry getEntry(String oauthToken) {
    return doGetEntry(oauthToken);
  }

  private OAuthEntry doGetEntry(String oauthToken) {
    return oAuthEntryService.getOAuthEntry(oauthToken);
  }

  @Override
  public SecurityToken getSecurityTokenForConsumerRequest(String consumerKey,
                                                          String userId, OAuthConsumer authConsumer) throws OAuthProblemException {
    String container = getEnvironment().getContainerName();
    return new OAuthSecurityToken(userId, null, consumerKey, DOMAIN, container,
            null, AuthenticationMode.OAUTH_CONSUMER_REQUEST.name(), null,
            (String) authConsumer.getProperty(OAuthSecurityToken.ENTITY_ID));
  }

  @Override
  public OAuthConsumer getConsumer(String consumerKey)
          throws OAuthProblemException {
    return doGetConsumer(consumerKey);
  }

  @SuppressWarnings("unchecked")
  private OAuthConsumer doGetConsumer(String consumerKey)
          throws OAuthProblemException {
    String url = getEnvironment().getBaseSocialApiUrl();

    /*
     * For possible logging purposed we define the json response here
     */
    String json = null;
    HashMap<String, Object> result;
    try {
      // https://engineblock-internal.dev.coin.surf.net/service/metadata?gadgeturl=http://localhost:8080/samplecontainer/examples/SocialHelloWorld.xml&keys=coin:oauth:secret
      url += "/service/metadata?gadgeturl="
              + URLEncoder.encode(consumerKey, "UTF-8")
              + "&keys="
              + URLEncoder
              .encode(
                      "coin:oauth:secret,coin:oauth:public_key,coin:oauth:app_title,coin:oauth:app_description,coin:oauth:app_thumbnail,coin:oauth:app_icon,coin:oauth:callback_url,"
                              + OAuthSecurityToken.ENTITY_ID, "UTF-8");
      json = IOUtils.toString(executeHttpGet(url));

      result = getObjectMapper().readValue(json, HashMap.class);

    } catch (Exception e) {

      String shortMessage = "Unexpected exception occured in retrieving Consumer";
      String formattedMessage = String.format(
              "Unexpected exception occured in retrieving Consumer while parsing (%s) for gadgetUrl "+consumerKey,
              json);
      String errorMessage = "An exception occured during the oAuth handling in the Shindig server.<br/><br/>" +
              getClass().getSimpleName() + "<br/>" + formattedMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage,
              errorMessage,
              e.getMessage(),
              getHost(),
              "Shindig");
      errorMail.setLocation(this.getClass().getName() + "#doGetConsumer");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new OAuthProblemException(shortMessage + " " + e.getMessage());
    }

    // We fetch some basic stuff from the result for validating if we have the
    // correct information.
    // For now an OAuth secret or a public key must be supplied
    // The callback Url is optional. If the callback Url isn't stored in the db
    // we create a consumer with an empty (null) callbackUrl
    String consumerSecret = (String) result.get("coin:oauth:secret");
    String callbackUrl = (String) result.get("coin:oauth:callback_url");

    // Do we have a public key?
    boolean isPublicKey = result.containsKey("coin:oauth:public_key");

    // If we neither have a secret nor a public key throw an exception. For, we
    // cannot create a valid consumer
    if (!StringUtils.hasText(consumerSecret) && !isPublicKey) {
      String shortMessage = "No secret nor public key for consumerKey  " +
              consumerKey;
      String errorMessage = "An exception occured during the oAuth handling in the Shindig server.<br/><br/>" +
              getClass().getSimpleName() + "<br/>" + shortMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage, errorMessage, "UNKNOWN", getHost(), "Shindig");
      errorMail.setLocation(this.getClass().getName() + "#doGetConsumer");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new OAuthProblemException(shortMessage);
    }

    OAuthConsumer consumer;
    // Create a consumer with a secret or a public key depending on the
    // information that is supplied.
    // We occasionally have a callback url in the db. If we do not have one,
    // the object will be null
    if (!isPublicKey) {
      consumer = new OAuthConsumer(callbackUrl, consumerKey, consumerSecret,
              SERVICE_PROVIDER);
    } else {
      consumer = new OAuthConsumer(callbackUrl, consumerKey, null,
              SERVICE_PROVIDER);
    }

    // Set public key if there is one in the results
    if (isPublicKey) {
      setCertificate(consumer, (String) result.get("coin:oauth:public_key"));
    }

    // Set additional properties. See the ImmutableList a the beginning of the
    // method to determine which attributes are set.
    // The properties are loosely based on the ModulePrefs of a gadget
    setAdditionalProperties(consumer, result);

    return consumer;
  }

  private String getHost() {
    try {
      return InetAddress.getLocalHost().toString();
    } catch (UnknownHostException e) {
      return "UNKNOWN";
    }
  }

  private void setAdditionalProperties(OAuthConsumer consumer,
                                       HashMap<String, Object> map) {
    for (String key : ADDITIONAL_PROPERTIES) {
      if (map.containsKey(key))
        consumer.setProperty(key, (String) map.get(key));
    }
  }

  /**
   * Create a Certificate from the supplied public key. For now we only support
   * X509 certificates that are base64 encoded.
   *
   * @param consumer  the {@link OAuthConsumer}
   * @param publicKey the base64 encode string containing the publicKey
   * @throws OAuthProblemException the exception thrown creating the certificate causes an
   *                               {@link CertificateException}
   */
  private void setCertificate(OAuthConsumer consumer, String publicKey)
          throws OAuthProblemException {
    byte[] buffer = Base64.decodeBase64(publicKey);
    try {
      CertificateFactory fac = CertificateFactory.getInstance("X509");
      ByteArrayInputStream in = new ByteArrayInputStream(buffer);
      X509Certificate cert = (X509Certificate) fac.generateCertificate(in);
      consumer.setProperty(RSA_SHA1.X509_CERTIFICATE, cert);
    } catch (CertificateException e) {
      throw new OAuthProblemException(
              "Could not create certificate from public key.");
    }
  }

  @Override
  public OAuthEntry generateRequestToken(String consumerKey,
                                         String oauthVersion, String signedCallbackUrl, String virtualOrganization)
          throws OAuthProblemException {
    return doGenerateRequestToken(consumerKey, oauthVersion, signedCallbackUrl,
            virtualOrganization);
  }

  private OAuthEntry doGenerateRequestToken(String consumerKey,
                                            String oauthVersion, String signedCallbackUrl, String virtualOrganization) {
    OAuthEntry entry = new OAuthEntry();
    entry.setAppId(consumerKey);
    entry.setConsumerKey(consumerKey);
    entry.setDomain(DOMAIN);
    entry.setContainer(getEnvironment().getContainerName());
    entry.setVirtualOrganization(virtualOrganization);

    entry.setToken(UUID.randomUUID().toString());
    entry.setTokenSecret(UUID.randomUUID().toString());

    entry.setType(OAuthEntry.Type.REQUEST);
    entry.setIssueTime(new Date());
    entry.setOauthVersion(oauthVersion);
    if (signedCallbackUrl != null) {
      entry.setCallbackUrlSigned(true);
      entry.setCallbackUrl(signedCallbackUrl);
    }

    // Save entry to database
    oAuthEntryService.saveOAuthEntry(entry);

    return entry;

  }

  @Override
  public OAuthEntry convertToAccessToken(OAuthEntry entry)
          throws OAuthProblemException {
    Preconditions.checkNotNull(entry);
    Preconditions.checkState(entry.getType() == OAuthEntry.Type.REQUEST,
            "Token must be a request token");

    OAuthEntry accessEntry = new OAuthEntry(entry);

    accessEntry.setToken(UUID.randomUUID().toString());
    accessEntry.setTokenSecret(UUID.randomUUID().toString());

    accessEntry.setType(OAuthEntry.Type.ACCESS);
    accessEntry.setIssueTime(new Date());

    oAuthEntryService.deleteOAuthEntry(entry.getToken());
    oAuthEntryService.saveOAuthEntry(accessEntry);

    return accessEntry;
  }

  @Override
  public void authorizeToken(OAuthEntry entry, String userId)
          throws OAuthProblemException {
    Preconditions.checkNotNull(entry);
    entry.setAuthorized(true);
    entry.setUserId(Preconditions.checkNotNull(userId));
    if (entry.isCallbackUrlSigned()) {
      entry.setCallbackToken(Crypto.getRandomDigits(CALLBACK_TOKEN_LENGTH));
    }
    oAuthEntryService.saveOAuthEntry(entry);
  }

  @Override
  public void disableToken(OAuthEntry entry) {
    Preconditions.checkNotNull(entry);
    entry.setCallbackTokenAttempts(entry.getCallbackTokenAttempts() + 1);
    if (!entry.isCallbackUrlSigned()
            || entry.getCallbackTokenAttempts() >= CALLBACK_TOKEN_ATTEMPTS) {
      entry.setType(OAuthEntry.Type.DISABLED);
    }

    oAuthEntryService.saveOAuthEntry(entry);
  }

  @Override
  public void removeToken(OAuthEntry entry) {
    Preconditions.checkNotNull(entry);
    oAuthEntryService.deleteOAuthEntry(entry.getToken());
  }

  public void setErrorMessageMailer(ErrorMessageMailer errorMessageMailer) {
    this.errorMessageMailer = errorMessageMailer;
  }
}