/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.oauth;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.HashMap;

import org.apache.commons.io.IOUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.gadgets.GadgetException;
import org.apache.shindig.gadgets.oauth.BasicOAuthStore;
import org.apache.shindig.gadgets.oauth.BasicOAuthStoreConsumerKeyAndSecret;
import org.apache.shindig.gadgets.oauth.BasicOAuthStoreConsumerKeyAndSecret.KeyType;
import org.apache.shindig.gadgets.oauth.OAuthStore;
import org.apache.shindig.social.core.oauth.OAuthSecurityToken;
import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.StringUtils;

import net.oauth.OAuth;
import net.oauth.OAuthConsumer;
import net.oauth.OAuthServiceProvider;
import net.oauth.signature.RSA_SHA1;
import nl.surfnet.coin.shared.domain.ErrorMail;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import nl.surfnet.coin.shindig.protocol.HttpClientProvider;
import nl.surfnet.coin.shindig.spi.TokenInfoService;

/**
 * {@link OAuthStore} that retrieves the consumer_key, consumer_secret and
 * key_type from the Engine Block
 * <p/>
 * Usage scenario: coin container is the oauth consumer (oauth data are stored elsewhere,
 * this container tries to fetch data from e.g. Google)
 */
public class CoinOAuthStore implements OAuthStore {// extends BasicOAuthStore {

  /**
   * Callback to use when no per-key callback URL is found.
   */
  private String defaultCallbackUrl;
  /*
   * For parsing the JSON
   */
  private static final ObjectMapper objectMapper = new ObjectMapper();

  private HttpClientProvider httpClientProvider = new HttpClientProvider();

  private TokenInfoService tokenInfoService;

  /*
   * Url for the engine block
   */
  private String baseUrl;
  /*
   * Private key for signing the RSH_PRIVATE oauth requests
   */
  private BasicOAuthStoreConsumerKeyAndSecret defaultKey;

  private ErrorMessageMailer errorMessageMailer;

    public CoinOAuthStore(String defaultCallbackUrl, String baseUrl,
                        TokenInfoService tokenInfoService, String pathToPrivateKey,
                        String privateKeyName, ErrorMessageMailer errorMessageMailer) throws IOException {
    super();
    this.baseUrl = baseUrl;
    this.defaultCallbackUrl = defaultCallbackUrl;
    this.tokenInfoService = tokenInfoService;
    this.defaultKey = loadDefaultKey(pathToPrivateKey, privateKeyName);
    this.errorMessageMailer = errorMessageMailer;
  }

  private BasicOAuthStoreConsumerKeyAndSecret loadDefaultKey(
          String signingKeyFile, String signingKeyName) throws IOException {
    InputStream inputStream = new ClassPathResource(signingKeyFile)
            .getInputStream();
    String privateKey = IOUtils.toString(inputStream);
    privateKey = BasicOAuthStore.convertFromOpenSsl(privateKey);
    return new BasicOAuthStoreConsumerKeyAndSecret(null, privateKey,
            KeyType.RSA_PRIVATE, signingKeyName, null);

  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.gadgets.oauth.BasicOAuthStore#getConsumerKeyAndSecret
   * (org.apache.shindig.auth.SecurityToken, java.lang.String,
   * net.oauth.OAuthServiceProvider)
   */
  @Override
  public ConsumerInfo getConsumerKeyAndSecret(SecurityToken securityToken,
                                              String serviceName, OAuthServiceProvider provider) throws GadgetException {
    try {
      String gadgetUrl = securityToken.getAppUrl();
      return doGetConsumerKeyAndSecret(gadgetUrl, serviceName, provider, securityToken);
    } catch (Exception e) {
      String shortMessage = "Exception in getConsumerKeyAndSecret";
      String errorMessage = "An exception occured during the oAuth handling in the Shindig server.<br/><br/>" +
              getClass().getSimpleName() + "<br/>" + shortMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage,
              errorMessage,
              e.getMessage(),
              getHost(),
              "Shindig");
      errorMail.setLocation(this.getClass().getName() + "#getConsumerKeyAndSecret");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new GadgetException(GadgetException.Code.INTERNAL_SERVER_ERROR,
              shortMessage, e);
    }
  }

  @SuppressWarnings("unchecked")
  private ConsumerInfo doGetConsumerKeyAndSecret(String gadgetUrl,
                                                 String serviceName, OAuthServiceProvider provider, SecurityToken securityToken) throws IOException,
          GadgetException {
    // https://engineblock-internal.dev.coin.surf.net/service/metadata?gadgeturl=http://localhost:8080/samplecontainer/examples/SocialHelloWorld.xml&keys=coin:oauth:secret,coin:oauth:consumer_key,coin:oauth:consumer_secret,coin:oauth:key_type
    String json = getOAuthJsonString(gadgetUrl);

    HashMap<String, Object> results;
    try {
      results = objectMapper.readValue(json, HashMap.class);
    } catch (Throwable t) {
      // we want to log the eb result, as this as caused much grievance
      String shortMessage = "Unexpected exception while parsing json from EB";
      String formattedMessage = "Unexpected exception while parsing json('" + json + "') from EB ";
      String errorMessage = "An exception occured during the oAuth handling in the Shindig server.<br/><br/>" +
              getClass().getSimpleName() + "<br/>" + formattedMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage,
              errorMessage,
              t.getMessage(),
              getHost(),
              "Shindig");
      errorMail.setLocation(this.getClass().getName() + "#doGetConsumerKeyAndSecret");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new GadgetException(GadgetException.Code.INTERNAL_SERVER_ERROR,
              errorMessage);
    }
    String consumerKey = (String) results.get("coin:oauth:consumer_key");
    String consumerSecret = (String) results.get("coin:oauth:consumer_secret");
    String keyType = (String) results.get("coin:oauth:key_type");
    String entityId = (String) results.get(OAuthSecurityToken.ENTITY_ID);

    if (!(StringUtils.hasText(consumerKey)
            && StringUtils.hasText(consumerSecret)
            && StringUtils.hasText(keyType) && StringUtils.hasText(entityId))) {

      String shortMessage = "no Key for gadget (" + gadgetUrl + ")";
      String formattedMessage = "No key for for gadget " + gadgetUrl + " and service " + serviceName + " and entityId " + entityId;
      String errorMessage = "An exception occured during the oAuth handling in the Shindig server.<br/><br/>" +
              getClass().getSimpleName() + "<br/>" + formattedMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage,
              errorMessage,
              "UNKNOWN",
              getHost(),
              "Shindig");
      errorMail.setLocation(this.getClass().getName() + "#doGetConsumerKeyAndSecret");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new GadgetException(GadgetException.Code.INTERNAL_SERVER_ERROR,
              errorMessage);
    }
    OAuthConsumer consumer;
    if ("RSA_PRIVATE".equals(keyType)) {
      consumer = new OAuthConsumer(null, consumerKey, null, provider);
      // The oauth.net java code has lots of magic. By setting this property
      // here, code thousands
      // of lines away knows that the consumerSecret value in the consumer
      // should be treated as
      // an RSA private key and not an HMAC key.
      consumer.setProperty(OAuth.OAUTH_SIGNATURE_METHOD, OAuth.RSA_SHA1);
      consumer.setProperty(RSA_SHA1.PRIVATE_KEY, defaultKey.getConsumerSecret());
    } else if ("HMAC_SHA1".equals(keyType)) {
      consumer = new OAuthConsumer(null, consumerKey, consumerSecret, provider);
      consumer.setProperty(OAuth.OAUTH_SIGNATURE_METHOD, OAuth.HMAC_SHA1);
    } else if ("PLAINTEXT".equals(keyType)) {
      consumer = new OAuthConsumer(null, consumerKey, consumerSecret, provider);
      consumer.setProperty(OAuth.OAUTH_SIGNATURE_METHOD, "PLAINTEXT");
    } else {
      String errorMessage = "Could not determine a suitable signature method (" + keyType + "). Only 'HMAC_SHA1', 'RSA_PRIVATE' or 'PLAINTEXT' are currently supported!";
      throw new GadgetException(GadgetException.Code.INTERNAL_SERVER_ERROR, errorMessage);
    }
    consumer.setProperty(OAuthSecurityToken.ENTITY_ID, entityId);
    return new ConsumerInfo(consumer, null, defaultCallbackUrl);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.gadgets.oauth.BasicOAuthStore#getTokenInfo(org.apache
   * .shindig.auth.SecurityToken,
   * org.apache.shindig.gadgets.oauth.OAuthStore.ConsumerInfo, java.lang.String,
   * java.lang.String)
   */
  @Override
  public TokenInfo getTokenInfo(SecurityToken securityToken,
                                ConsumerInfo consumerInfo, String serviceName, String tokenName) {
    TokenInfo tokenInfo = null;
    CoinOAuthTokenInfo info = tokenInfoService.getCoinOAuthTokenInfo(
            securityToken.getViewerId(), securityToken.getAppUrl(),
            CoinOAuthTokenInfo.MODULE_ID, tokenName, serviceName);
    if (info != null) {
      tokenInfo = new TokenInfo(info.getAccessToken(), info.getTokenSecret(),
              info.getSessionHandle(), info.getTokenExpireMillis());
    }
    return tokenInfo;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.gadgets.oauth.BasicOAuthStore#setTokenInfo(org.apache
   * .shindig.auth.SecurityToken,
   * org.apache.shindig.gadgets.oauth.OAuthStore.ConsumerInfo, java.lang.String,
   * java.lang.String, org.apache.shindig.gadgets.oauth.OAuthStore.TokenInfo)
   */
  @Override
  public void setTokenInfo(SecurityToken securityToken,
                           ConsumerInfo consumerInfo, String serviceName, String tokenName,
                           TokenInfo tokenInfo) {
    // we store the token in the database
    CoinOAuthTokenInfo info = new CoinOAuthTokenInfo(securityToken,
            serviceName, tokenName, tokenInfo);
    tokenInfoService.saveCoinOAuthTokenInfo(info);
  }

  @Override
  public void removeToken(SecurityToken securityToken,
                          ConsumerInfo consumerInfo, String serviceName, String tokenName)
          throws GadgetException {
    tokenInfoService.deleteCoinOAuthTokenInfo(
            securityToken.getViewerId(), securityToken.getAppUrl(),
            CoinOAuthTokenInfo.MODULE_ID, tokenName, serviceName);
  }

  /**
   * Returns the oAuth Json string for the gadget
   *
   * @param gadgetUrl the location of the gadgetdefinition
   * @return Json string
   * @throws IOException     if the oAuth info cannot be fetched
   * @throws GadgetException in case of an empty response
   */
  private String getOAuthJsonString(String gadgetUrl) throws IOException, GadgetException {
    String url = baseUrl
            + "/service/metadata?gadgeturl="
            + URLEncoder.encode(gadgetUrl, "UTF-8")
            + "&keys="
            + URLEncoder.encode(
            "coin:oauth:consumer_key,coin:oauth:consumer_secret,coin:oauth:key_type,entityId",
            "UTF-8");
    InputStream inputStream = httpClientProvider.getHttpClient()
            .execute(new HttpGet(url)).getEntity().getContent();

    String json;
    try {
      json = IOUtils.toString(inputStream);
    } finally {
      inputStream.close();
    }
    if (!StringUtils.hasText(json)) {
      String shortMessage = String.format("Url '%s' returned empty result", url);
      String errorMessage = "An exception occured during the oAuth handling in the Shindig server.<br/><br/>" +
              getClass().getSimpleName() + "<br/>" + shortMessage;
      ErrorMail errorMail = new ErrorMail(shortMessage,
              errorMessage,
              "UNKNOWN",
              getHost(),
              "Shindig");
      errorMessageMailer.sendErrorMail(errorMail);
      throw new GadgetException(GadgetException.Code.EMPTY_XML_DOCUMENT, shortMessage);
    }
    return json;
  }

  private String getHost() {
    try {
      return InetAddress.getLocalHost().toString();
    } catch (UnknownHostException e) {
      return "UNKNOWN";
    }
  }
}
