/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.oauth;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import nl.surfnet.coin.shindig.spi.CoinGuiceModule;
import nl.surfnet.coin.shindig.spi.TokenInfoService;
import org.apache.shindig.gadgets.oauth.OAuthStore;

import java.io.IOException;

/**
 * 
 * {@link Singleton} {@link Provider} for {@link CoinOAuthStore}
 * 
 */
@Singleton
public class CoinOAuthStoreProvider implements Provider<OAuthStore> {

  private static final String OAUTH_CALLBACK_URL = "shindig.signing.global-callback-url";
  private static final String ENGINE_BLOCK_BASE_URL = "base-social-api-url";
  private static final String OAUTH_SIGNING_KEY_FILE = "shindig.signing.key-file";
  private static final String OAUTH_SIGNING_KEY_NAME = "shindig.signing.key-name";

  private final CoinOAuthStore store;

  @Inject
  public CoinOAuthStoreProvider(
      @Named(OAUTH_CALLBACK_URL) String defaultCallbackUrl,
      @Named(ENGINE_BLOCK_BASE_URL) String baseUrl,
      @Named(OAUTH_SIGNING_KEY_FILE) String pathToPrivateKey,
      @Named(OAUTH_SIGNING_KEY_NAME) String privateKeyName
      ) throws IOException {
    super();
    this.store = new CoinOAuthStore(defaultCallbackUrl, baseUrl,
        getTokenInfoService(), pathToPrivateKey, privateKeyName, getErrorMessageMailer());
  }

  protected ErrorMessageMailer getErrorMessageMailer() {
    return CoinGuiceModule.getContext().getBean("errorMessageMailer", ErrorMessageMailer.class);
  }
  
  protected TokenInfoService getTokenInfoService() {
    return CoinGuiceModule.getContext().getBean("tokenInfoService",
        TokenInfoService.class);
  }

  @Override
  public OAuthStore get() {
    return store;
  }

}
