/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * 
 */
package nl.surfnet.coin.shindig.shiro;

import java.io.IOException;
import java.net.URLEncoder;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import nl.surfnet.coin.shindig.oauth.CoinOAuthServlet;
import nl.surfnet.coin.shindig.protocol.ContextEnvironment;
import nl.surfnet.coin.shindig.spi.CoinGuiceModule;
import nl.surfnet.coin.shindig.spi.OAuthEntryService;

import org.apache.shindig.social.opensocial.oauth.OAuthEntry;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.subject.Subject;
import org.apache.shiro.web.filter.authc.AuthenticatingFilter;
import org.apache.shiro.web.filter.authc.FormAuthenticationFilter;
import org.apache.shiro.web.util.SavedRequest;
import org.apache.shiro.web.util.WebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

/**
 * @author steinwelberg
 * 
 */
public class CoinAuthenticationFilter extends AuthenticatingFilter {

  public static final String DEFAULT_ERROR_KEY_ATTRIBUTE_NAME = "shiroLoginFailure";
  private static final String PASSWORD = "password";

  private String failureKeyAttribute = DEFAULT_ERROR_KEY_ATTRIBUTE_NAME;
  private ContextEnvironment environment;

  private static final Logger logger = LoggerFactory
      .getLogger(FormAuthenticationFilter.class);

  public CoinAuthenticationFilter() {
    setLoginUrl(DEFAULT_LOGIN_URL);
  }

  @Override
  public void setLoginUrl(String loginUrl) {
    String previous = getLoginUrl();
    if (previous != null) {
      this.appliedPaths.remove(previous);
    }
    super.setLoginUrl(loginUrl);
    if (logger.isDebugEnabled()) {
      logger.debug("Adding login url to applied paths.");
    }
    this.appliedPaths.put(getLoginUrl(), null);
  }

  public String getFailureKeyAttribute() {
    return failureKeyAttribute;
  }

  public void setFailureKeyAttribute(String failureKeyAttribute) {
    this.failureKeyAttribute = failureKeyAttribute;
  }

  /**
   * This method creates the authentication token with the user name & password.
   * 
   * @param request
   *          the {@link HttpServletRequest}
   * @param response
   *          the {@link HttpServletResponse}
   * 
   */
  @Override
  protected AuthenticationToken createToken(ServletRequest request,
      ServletResponse response) throws Exception {
    String remoteUser = getRemoteUser(request);

    // Remote user is empty, user is not logged in.
    if (!StringUtils.hasText(remoteUser)) {
      return null;
    } else {
      // If user is logged in create token.
      return createToken(remoteUser, PASSWORD, request, response);
    }
  }

  /**
   * The method that is called when an access request is denied because for
   * instance a user is not logged in. This method tries to determine whether
   * the login form is submitted and then calls the <code>executeLogin</code>
   * method.
   * 
   * @param request
   *          the {@link ServletRequest}
   * @param response
   *          the {@link ServletResponse}
   */
  @Override
  protected boolean onAccessDenied(ServletRequest request,
      ServletResponse response) throws Exception {
    if (isLoginRequest(request, response)) {
      if (isLoginSubmission(request, response)) {
        String remoteUser = getRemoteUser(request);
        if (!StringUtils.hasText(remoteUser)) {
          redirectUserToShibboleth(request, response);
          return false;
        } else {
          if (logger.isDebugEnabled()) {
            logger
                .debug("Login submission detected.  Attempting to execute login.");
          }
          return executeLogin(request, response);
        }
      } else {
        if (logger.isDebugEnabled()) {
          logger.trace("Login page view.");
        }
        // allow them to see the login page ;)
        return true;
      }
    } else {
      if (logger.isDebugEnabled()) {
        logger
            .debug("Attempting to access a path which requires authentication.  Forwarding to the "
                + "Authentication url [" + getLoginUrl() + "]");
      }

      saveRequestAndRedirectToLogin(request, response);
      return false;
    }
  }

  /**
   * Redirect to shibboleth for login handling. After successful login the user
   * is returned to his original request url for proper Shiro login handling.
   * 
   * @param request
   *          the {@link ServletRequest}
   * @param response
   *          the {@link ServletResponse}
   * @throws IOException
   */
  private void redirectUserToShibboleth(ServletRequest request,
      ServletResponse response) throws IOException {
    HttpServletResponse httpResponse = (HttpServletResponse) response;

    ContextEnvironment environment = getEnvironment();

    // Get login Url
    String shindigHost = environment.getShindigBaseUrl();
    String shibLoginUrlSuffix = environment
        .getShibbolethAuthenticationUrlSuffix();
    String target = URLEncoder.encode(shindigHost + "login.jsp?login=true",
        "UTF-8");

    // are we in a VO context?
    String virtualOrganization = getVirtualOrganization(request);
    String entityID = "";
    if (StringUtils.hasText(virtualOrganization)) {
      entityID = "&entityID="
          + URLEncoder.encode(getEnvironment().getVoMetadataPrefix()
              + virtualOrganization, "UTF-8");
    }
    // Send user to shibboleth for login.
    String location = shindigHost + shibLoginUrlSuffix + "?target=" + target
        + entityID;
    httpResponse.sendRedirect(location);
  }

  /*
   * Tricky, we need to get the RequestToken to check the virtualOrganization.
   * The only place to retrieve it is from the RequestToken. Luckily we have
   * access to the oauth_token from the referer.
   */
  private String getVirtualOrganization(ServletRequest request) {
    HttpServletRequest servletRequest = (HttpServletRequest) request;
    HttpSession session = servletRequest.getSession(false);
    String virtualOrganization = null;
    if (session != null) {
      SavedRequest savedRequest = (SavedRequest) session
          .getAttribute("shiroSavedRequest");
      if (savedRequest != null) {
        String query = savedRequest.getQueryString();
        if (query != null) {
          int indexOf = query.indexOf("oauth_token");
          if (indexOf > -1) {
            int endIndex = query.indexOf("&", indexOf);
            endIndex = (endIndex < 0 ? query.length() : endIndex);
            String token = query.substring(indexOf + "oauth_token=".length(),
                endIndex);
            OAuthEntryService entryService = (OAuthEntryService) CoinGuiceModule
                .getContext().getBean("oAuthEntryService");
            OAuthEntry oAuthEntry = entryService.getOAuthEntry(token);
            if (oAuthEntry != null) {
              virtualOrganization = oAuthEntry.getVirtualOrganization();
            }
          }
        }
      }
    }
    return virtualOrganization;
  }

  private ContextEnvironment getEnvironment() {
    if (this.environment == null) {
      this.environment = (ContextEnvironment) CoinGuiceModule.getContext()
          .getBean("environment");
    }
    return environment;
  }

  /**
   * The method that is called on login success, so that the issuer can go to
   * the original page
   * 
   * @param token
   *          the {@link AuthenticationToken} for this successful login attempt
   * @param subject
   *          the {@link Subject} (logged in user)
   * @param request
   *          the {@link ServletRequest}
   * @param response
   *          the {@link ServletResponse}
   */
  protected boolean onLoginSuccess(AuthenticationToken token, Subject subject,
      ServletRequest request, ServletResponse response) throws Exception {
    issueSuccessRedirect(request, response);
    // we handled the success redirect directly, prevent the chain from
    // continuing:
    return false;
  }

  /**
   * The function that is triggerd on login Failure
   * 
   * @param token
   *          the {@link AuthenticationToken}
   * @param e
   *          the {@link AuthenticationException} that was thrown
   * @param request
   *          the {@link ServletRequest} for this request.
   * @param response
   *          the {@link ServletResponse} for this request.
   */
  protected boolean onLoginFailure(AuthenticationToken token,
      AuthenticationException e, ServletRequest request,
      ServletResponse response) {
    setFailureAttribute(request, e);
    // login failed, let request continue back to the login page:
    return true;
  }

  /**
   * This default implementation merely returns <code>true</code> if the request
   * is an HTTP <code>POST</code>, <code>false</code> otherwise. Can be
   * overridden by subclasses for custom login submission detection behavior.
   * 
   * @param request
   *          the incoming ServletRequest
   * @param response
   *          the outgoing ServletResponse.
   * @return <code>true</code> if the request is an HTTP <code>POST</code>,
   *         <code>false</code> otherwise.
   */
  protected boolean isLoginSubmission(ServletRequest request,
      ServletResponse response) {
    return (request instanceof HttpServletRequest)
        && WebUtils.toHttp(request).getMethod().equalsIgnoreCase(POST_METHOD);
  }

  /**
   * Set the reason for the authentication failure
   * 
   * @param request
   *          the {@link HttpServletRequest} for the request that was issued.
   * @param ae
   *          the {@link AuthenticationException} that was thrown.
   */
  protected void setFailureAttribute(ServletRequest request,
      AuthenticationException ae) {
    String className = ae.getClass().getName();
    request.setAttribute(getFailureKeyAttribute(), className);
  }

  /**
   * Hook for subclasses to override the shibboleth default behaviour
   * 
   * @param request
   *          the httpRequest
   * @return the String of the logged in user
   */
  protected String getRemoteUser(ServletRequest request) {
    String shibbolethDummyUser = getEnvironment().getShibbolethDummyUser();
    if (StringUtils.hasText(shibbolethDummyUser)) {
      return shibbolethDummyUser;
    }
    HttpServletRequest httpRequest = (HttpServletRequest) request;
    return httpRequest.getHeader("REMOTE_USER");
  }

}
