/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;

import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.util.ImmediateFuture;
import org.apache.shindig.protocol.DataCollection;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.social.opensocial.spi.AppDataService;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.hibernate.Query;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import nl.surfnet.coin.shindig.model.PersonAppData;

/**
 * Implementation of {@link AppDataService}
 * 
 */
@Component(value = "appDataService")
@Transactional
public class AppDataServiceImpl implements AppDataService {

  @Autowired
  private OpenSocialValidator openSocialValidator;

  @Autowired
  private SessionFactory sessionFactory;

  /*
   * Hibernate class
   */
  private static final String PERSON_APP_DATA = PersonAppData.class.getName();

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.AppDataService#deletePersonData
   * (org.apache.shindig.social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.spi.GroupId, java.lang.String,
   * java.util.Set, org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<Void> deletePersonData(UserId userId, GroupId groupId,
      String appId, Set<String> fields, SecurityToken token)
      throws ProtocolException {
    openSocialValidator.invariant(userId);
    StringBuilder queryString = new StringBuilder("delete from "
        + PERSON_APP_DATA + " where personId = :personId");
    if (StringUtils.hasText(appId)) {
      queryString.append(" and appId = :appId");
    }
    Query query = sessionFactory.getCurrentSession()
        .createQuery(queryString.toString())
        .setString("personId", userId.getUserId(token));
    if (StringUtils.hasText(appId)) {
      query.setParameter("appId", appId);
    }
    query.executeUpdate();
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.AppDataService#getPersonData(java
   * .util.Set, org.apache.shindig.social.opensocial.spi.GroupId,
   * java.lang.String, java.util.Set, org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<DataCollection> getPersonData(Set<UserId> userIds,
      GroupId groupId, String appId, Set<String> fields, SecurityToken token)
      throws ProtocolException {
    openSocialValidator.invariant(userIds, true);
    List<PersonAppData> result = getPersonData(userIds, appId, token);
    return constructDataCollection(result);

  }

  /*
   * Get PersonAppData
   */
  @SuppressWarnings("unchecked")
  private List<PersonAppData> getPersonData(Set<UserId> userIds, String appId,
      SecurityToken token) {
    StringBuilder queryString = new StringBuilder("from " + PERSON_APP_DATA
        + " where personId in (:userIds) ");
    if (StringUtils.hasText(appId)) {
      queryString.append(" and appId = :appId");
    }
    Query query = sessionFactory.getCurrentSession().createQuery(
        queryString.toString());
    query.setParameter("userIds", userIds.iterator().next().getUserId(token));
    if (StringUtils.hasText(appId)) {
      query.setParameter("appId", appId);
    }
    return query.list();
  }

  /*
   * Wrap the personAppData in a DataCollection
   */
  private Future<DataCollection> constructDataCollection(
      List<PersonAppData> result) {
    Map<String, Map<String, String>> data = new HashMap<String, Map<String, String>>();
    for (PersonAppData personAppData : result) {
      // in order to avoid lazy loading exceptions with Hibernate we clone the
      // map
      data.put(personAppData.getAppId(), cloneData(personAppData.getData()));
    }
    return ImmediateFuture.newInstance(new DataCollection(data));
  }

  private Map<String, String> cloneData(Map<String, String> data) {
    Map<String, String> result = new HashMap<String, String>();
    if (data != null) {
      Set<String> keySet = data.keySet();
      for (String key : keySet) {
        result.put(key, data.get(key));
      }
    }
    return result;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.AppDataService#updatePersonData
   * (org.apache.shindig.social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.spi.GroupId, java.lang.String,
   * java.util.Set, java.util.Map, org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<Void> updatePersonData(UserId userId, GroupId groupId,
      String appId, Set<String> fields, Map<String, String> values,
      SecurityToken token) throws ProtocolException {
    openSocialValidator.invariant(userId);
    openSocialValidator.invariant(null, null, appId);
    List<PersonAppData> result = getPersonData(Collections.singleton(userId),
        appId, token);
    PersonAppData appData;
    if (CollectionUtils.isEmpty(result)) {
      appData = new PersonAppData();
      appData.setAppId(appId);
      appData.setData(values);
      appData.setPersonId(userId.getUserId(token));
    } else {
      appData = result.get(0);
      appData.getData().putAll(values);
    }
    sessionFactory.getCurrentSession().saveOrUpdate(appData);
    return null;
  }

  /**
   * @param openSocialValidator
   *          the openSocialId to set
   */
  public void setOpenSocialId(OpenSocialValidator openSocialValidator) {
    this.openSocialValidator = openSocialValidator;
  }

  /**
   * @param sessionFactory
   *          the sessionFactory to set
   */
  public void setSessionFactory(SessionFactory sessionFactory) {
    this.sessionFactory = sessionFactory;
  }

}
