/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import com.google.inject.AbstractModule;
import com.google.inject.name.Names;

import org.apache.shindig.social.opensocial.oauth.OAuthDataStore;
import org.apache.shindig.social.opensocial.spi.ActivityService;
import org.apache.shindig.social.opensocial.spi.AlbumService;
import org.apache.shindig.social.opensocial.spi.AppDataService;
import org.apache.shindig.social.opensocial.spi.GroupService;
import org.apache.shindig.social.opensocial.spi.MediaItemService;
import org.apache.shindig.social.opensocial.spi.MessageService;
import org.apache.shindig.social.opensocial.spi.PersonService;
import org.apache.shindig.social.sample.oauth.SampleOAuthDataStore;
import org.apache.shindig.social.sample.spi.JsonDbOpensocialService;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import nl.surfnet.coin.shindig.protocol.ContextEnvironment;

/**
 * {@link AbstractModule} that binds the custom implementation of the Shindig
 * spi interfaces. It is used in the web.xml of the shindig-server war. It uses
 * Spring to lookup the dependencies
 * 
 */
public class CoinGuiceModule extends AbstractModule {

  private static AbstractApplicationContext context;

  /**
   * Constructor
   */
  public CoinGuiceModule() {
    super();
    CoinGuiceModule.context = new ClassPathXmlApplicationContext(
        "coin-shindig-context.xml", "coin-shindig-properties-context.xml", "coin-shared-context.xml");
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.google.inject.AbstractModule#configure()
   */
  @Override
  protected void configure() {
    /*
     * This is needed for switching between modes (the value is configured in
     * the coin-portal.properties
     */
    ContextEnvironment environment = (ContextEnvironment) context
        .getBean("environment");
    if (environment.isDummyMode()) {
      configureDummy();
    } else {
      configureReal();

    }

  }

  /**
   * We bind with the default test bindings from shindig
   */
  private void configureDummy() {
    bind(String.class).annotatedWith(Names.named("shindig.canonical.json.db"))
        .toInstance("sampledata/canonicaldb.json");
    bind(ActivityService.class).to(JsonDbOpensocialService.class);
    bind(AlbumService.class).to(JsonDbOpensocialService.class);
    bind(MediaItemService.class).to(JsonDbOpensocialService.class);
    bind(AppDataService.class).to(JsonDbOpensocialService.class);
    bind(PersonService.class).to(JsonDbOpensocialService.class);
    bind(MessageService.class).to(JsonDbOpensocialService.class);
    bind(GroupService.class).to(JsonDbOpensocialService.class);
    bind(OAuthDataStore.class).to(SampleOAuthDataStore.class);
  }

  /**
   * We bind with the Coin implementation
   */
  private void configureReal() {
    bind(String.class).annotatedWith(Names.named("shindig.canonical.json.db"))
        .toInstance("sampledata/canonicaldb.json");
    bind(AlbumService.class).to(JsonDbOpensocialService.class);
    bind(MediaItemService.class).to(JsonDbOpensocialService.class);
    bind(ActivityService.class).toInstance(
        (ActivityService) context.getBean("activityService"));
    bind(AppDataService.class).toInstance(
        (AppDataService) context.getBean("appDataService"));
    bind(PersonService.class).toInstance(
        (PersonService) context.getBean("personService"));
    bind(MessageService.class).toInstance(
        (MessageService) context.getBean("messageService"));
    bind(GroupService.class).toInstance(
        (GroupService) context.getBean("groupService"));
    bind(OAuthDataStore.class).toInstance(
        (OAuthDataStore) context.getBean("oAuthDataStore"));
  }

  /**
   * @return the context
   */
  public static AbstractApplicationContext getContext() {
    return context;
  }

 
}
