/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;

import nl.surfnet.coin.shared.domain.ErrorMail;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;

import org.apache.commons.lang.StringUtils;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.util.ImmediateFuture;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.social.core.model.GroupImpl;
import org.apache.shindig.social.opensocial.model.Group;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.GroupId.Type;
import org.apache.shindig.social.opensocial.spi.GroupService;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * Implementation of shindig {@link GroupService} connecting to the EngineBlock
 * PHP endpoint
 */
@Component(value = "groupService")
public class GroupServiceImpl extends AbstractRestDelegationService implements
    GroupService {
  /*
   * Extension on the baseSocialApiUrl for the people functionality
   */
  private static final String GROUP_URL = "/social/groups/";

  @Autowired
  private ErrorMessageMailer errorMessageMailer;

  private Future<RestfulCollection<Group>> doGetGroups(String idOfUser,
      CollectionOptions options, SecurityToken token) throws IOException {
    StringBuilder url = new StringBuilder(getEnvironment()
        .getBaseSocialApiUrl() + GROUP_URL + idOfUser + "?1=1");

    String vo = super.getVirtualOrganizationContext(token, options);
    if (!StringUtils.isBlank(vo)) {
      url.append("&vo=" + vo);
    }
    String serviceProviderContext = super.getServiceProviderContext(token);
    if (!StringUtils.isBlank(serviceProviderContext)) {
      url.append("&sp-entity-id="
          + URLEncoder.encode(serviceProviderContext, UTF_8));
    }
    InputStream inputStream = executeHttpGet(url.toString());
    List<GroupImpl> constructGroups = constructGroups(inputStream);
    inputStream.close();
    List<Group> groups = new ArrayList<Group>();
    if (constructGroups != null) {
      groups.addAll(constructGroups);
    }
    return ImmediateFuture.newInstance(new RestfulCollection<Group>(groups));
  }

  @SuppressWarnings("unchecked")
  private List<GroupImpl> constructGroups(InputStream inputStream) {
    try {
      HashMap<String, Object> result = getObjectMapper().readValue(inputStream,
          HashMap.class);
      List entries = (List) result.get("entry");
      if (CollectionUtils.isEmpty(entries) || entries.get(0) instanceof List) {
        return null;
      }
      List<Map> entry = entries;
      List<GroupImpl> groupList = new ArrayList<GroupImpl>();
      for (Map groupMap : entry) {
        String id = (String) groupMap.get("id");
        String description = (String) groupMap.get("description");
        String title = (String) groupMap.get("title");
        String voot_membership_role = (String) groupMap
            .get("voot_membership_role");
        GroupImpl group = new GroupImpl();
        group.setDescription(description);
        group.setId(new GroupId(Type.groupId, id));
        group.setTitle(title);
        group.setVoot_membership_role(voot_membership_role);
        groupList.add(group);
      }

      return groupList;
    } catch (Exception e) {
      // matter of contract
      return null;
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.apache.shindig.social.opensocial.spi.GroupService#getGroups(org.apache
   * .shindig.social.opensocial.spi.UserId,
   * org.apache.shindig.social.opensocial.spi.CollectionOptions, java.util.Set,
   * org.apache.shindig.auth.SecurityToken)
   */
  @Override
  public Future<RestfulCollection<Group>> getGroups(UserId userId,
      CollectionOptions options, Set<String> fields, SecurityToken token) {
    try {
      String idOfUser = userId.getUserId(token);
      String onBehalfOf = (token != null ? token.getOwnerId() : null);
      if (StringUtils.isNotBlank(onBehalfOf) && !idOfUser.equals(onBehalfOf)) {
        throw new IllegalArgumentException(
            "Retrieval groups not granted as this " + idOfUser + " is not "
                + onBehalfOf);
      }
      this.logApiCall(token);
      return doGetGroups(idOfUser, options, token);
    } catch (Exception e) {
      String shortMessage = "Exception in getGroups(userId='"
          + userId.getUserId(token) + "')";
      String errorMessage = "An exception occured in the GroupService in the Shindig server\n"
          + "connecting to the EngineBlock PHP endpoint:<br/><br/>"
          + getClass().getSimpleName() + "<br/>" + shortMessage;
      ErrorMail errorMailDetails = new ErrorMail(shortMessage, errorMessage,
          userId.getUserId(token), e.getMessage(), getHost(), "Shindig");

      errorMessageMailer.sendErrorMail(errorMailDetails);
      throw new RuntimeException(shortMessage, e);
    }
  }

  private String getHost() {
    try {
      return InetAddress.getLocalHost().toString();
    } catch (UnknownHostException e) {
      return "UNKNOWN";
    }
  }

  public void setErrorMessageMailer(ErrorMessageMailer errorMessageMailer) {
    this.errorMessageMailer = errorMessageMailer;
  }
}
