/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.mock;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.social.core.model.ActivityImpl;
import org.apache.shindig.social.core.model.GroupImpl;
import org.apache.shindig.social.opensocial.model.Activity;
import org.apache.shindig.social.opensocial.spi.ActivityService;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.GroupId.Type;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.hibernate.SessionFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.internal.stubbing.answers.Returns;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.transaction.TransactionConfiguration;
import org.springframework.transaction.annotation.Transactional;

import nl.surfnet.coin.mock.AbstractMockHttpServerTest;

/**
 * {@link Test} for {@link ActivityServiceImpl}
 * 
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = {
    "classpath:coin-shindig-hsqldb-properties-context.xml",
    "classpath:coin-shindig-context.xml",
    "classpath:coin-shared-context.xml"})
@TransactionConfiguration(transactionManager = "transactionManager", defaultRollback = true)
@Transactional
public class ActivityServiceImplTest extends AbstractMockHttpServerTest {

  @Autowired
  private ActivityService activityService;

  @Autowired
  private CoinActivityService coinActivityService;

  @Autowired
  private SessionFactory sessionFactory;

  private String appId1 = "testAppId";
  private String appId2 = "unknown";
  private UserId user1 = new UserId(UserId.Type.userId, "user1");
  private GroupId groupId = new GroupId(Type.groupId, "testGroupId");
  private String activityId;

  @Before
  public void setupDbContent() throws Exception {
    Activity activity = new ActivityImpl();
    activity.setAppId(appId1);
    activity.setBody("body");
    setResponseResource(new ClassPathResource("json/mock-single-person.json"));
    activityService.createActivity(user1, new GroupId(GroupId.Type.groupId,
        groupId.getGroupId()), appId1, null, activity, getMockToken(user1));
    activityId = activity.getId();

    activity = new ActivityImpl();
    activity.setAppId(appId2);
    activity.setBody("body2");
    activityService.createActivity(user1, new GroupId(GroupId.Type.groupId,
        groupId.getGroupId()), appId2, null, activity, getMockToken(user1));

    sessionFactory.getCurrentSession().flush();
    
    
  }

  @After
  public void flushSession() {
    // flushing forces the actual sql to be executed
    sessionFactory.getCurrentSession().flush();
  }

  /**
   * Test method for
   * {@link nl.surfnet.coin.shindig.spi.ActivityServiceImpl#createActivity(org.apache.shindig.social.opensocial.spi.UserId, org.apache.shindig.social.opensocial.spi.GroupId, java.lang.String, java.util.Set, org.apache.shindig.social.opensocial.model.Activity, org.apache.shindig.auth.SecurityToken)}
   * .
   */
  @Test
  public void testGetActivity() throws Exception {
    Activity activity = activityService.getActivity(user1, null, null, null,
        activityId, getMockToken(user1)).get();
    assertEquals("body", activity.getBody());

  }

  @Test
  public void testGetActivitiesByAppId() throws Exception {
    Collection<Activity> activities = activityService
        .getActivities(new HashSet<UserId>(), groupId, appId2, null, null, getMockToken(user1))
        .get().getEntry();
    assertEquals(1, activities.size());

    activities = activityService
        .getActivities(Collections.singleton(user1), null, null, null, null,
            null).get().getEntry();
    assertEquals(0, activities.size());
  }

  @Test
  public void testFindActivityByGroupId() throws Exception {
    GroupImpl group = new GroupImpl();
    group.setId(groupId);
    List<Activity> activities = coinActivityService.findActivityByGroups(group,
        1, 1);
    assertEquals(1, activities.size());
  }
  
  /**
   * @param id
   * @return
   */
  private SecurityToken getMockToken(UserId id) {
    SecurityToken mock = mock(SecurityToken.class, new Returns(id.getUserId()));
    return mock;
  }

}
