/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.shindig.spi;

import nl.surfnet.coin.mock.AbstractMockHttpServerTest;
import nl.surfnet.coin.shared.service.ErrorMessageMailer;
import nl.surfnet.coin.shindig.protocol.ContextEnvironment;
import nl.surfnet.coin.shindig.protocol.HttpClientProvider;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.social.core.oauth.OAuthSecurityToken;
import org.apache.shindig.social.opensocial.model.Group;
import org.apache.shindig.social.opensocial.spi.UserId;
import org.apache.shindig.social.opensocial.spi.UserId.Type;
import org.junit.Assert;
import org.junit.Test;
import org.springframework.core.io.ClassPathResource;

import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Mockito.mock;

/**
 * {@link Test} for {@link GroupServiceImpl}
 */
public class GroupServiceImplTest extends AbstractMockHttpServerTest {

  /*
   * Holds no state
   */
  private GroupServiceImpl groupService;

  /**
   * Constructor
   */
  public GroupServiceImplTest() {
    super();
    groupService = new GroupServiceImpl();
    ContextEnvironment environment = new ContextEnvironment();
    environment.setBaseSocialApiUrl("http://localhost:8088/social/groups");
    groupService.setEnvironment(environment);
    groupService.setHttpClientProvider(new HttpClientProvider());
    groupService.setErrorMessageMailer(new ErrorMessageMailer());
    groupService.setLogService(new ApiCallLogServiceMock());
  }

  /**
   * Test method for
   * {@link GroupServiceImpl#getGroups(org.apache.shindig.social.opensocial.spi.UserId, org.apache.shindig.social.opensocial.spi.CollectionOptions, java.util.Set, org.apache.shindig.auth.SecurityToken)}
   * .
   */
  @Test
  public void testGetPersonGroups() throws Exception {
    UserId id = new UserId(Type.userId, "urn:collab:person:surfnet.nl:hansz");
    SecurityToken token = mock(SecurityToken.class);
    setResponseResource(new ClassPathResource("json/mock-multiple-teams.json"));
    List<Group> groups = groupService.getGroups(id, null, null, token).get()
            .getEntry();
    assertEquals(4, groups.size());
    for (Group group : groups) {
      assertNotNull(group.getTitle());
    }

    setResponseResource(new ClassPathResource("json/mock-single-team.json"));
    groups = groupService.getGroups(id, null, null, token).get().getEntry();
    assertEquals(1, groups.size());

    setResponseResource(new ClassPathResource("json/mock-empty-team.json"));
    groups = groupService.getGroups(id, null, null, token).get().getEntry();
    assertEquals(0, groups.size());
  }

  /**
   * Test method for
   * {@link GroupServiceImpl#getGroups(org.apache.shindig.social.opensocial.spi.UserId, org.apache.shindig.social.opensocial.spi.CollectionOptions, java.util.Set, org.apache.shindig.auth.SecurityToken)}
   * .
   */
  @Test
  public void testGetPersonGroupsSecurityException() throws Exception {
    UserId id = new UserId(Type.userId, "personId");
    SecurityToken token = new OAuthSecurityToken("onBehalfOf", "appUrl",
            "appId", "domain", "container", Long.MAX_VALUE, "authMode", /* virtualOrganization */
            null, /* serviceProviderEntityId */null);
    setResponseResource(new ClassPathResource("json/mock-multiple-teams.json"));
    try {
      groupService.getGroups(id, null, null, token).get().getEntry();
      Assert.fail("IllegalArgumentException because of security breach ");
    } catch (RuntimeException e) {
      String message = e.getCause().getMessage();
      Assert.assertTrue(message.contains("not granted"));
    }

  }

}
