/*
 * Copyright 2011 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.testsp.control;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nl.surfnet.coin.testsp.interceptor.LoginInterceptor;
import nl.surfnet.coin.testsp.service.OAuthServiceConsumer;
import nl.surfnet.coin.testsp.service.OpenSocialService;
import nl.surfnet.coin.testsp.util.Environment;

import org.opensocial.models.Group;
import org.opensocial.models.Person;
import org.opensocial.providers.Provider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;

/**
 * {@link Controller} that handles the "home page" of the testsp
 */
@Controller
public class HomeController {

  private static final Logger logger = LoggerFactory
      .getLogger(HomeController.class);

  private static final String VIRTUAL_ORGANIZATION = "VIRTUAL_ORGANIZATION";

  @Autowired
  private OAuthServiceConsumer serviceConsumer;

  @Autowired
  private OpenSocialService openSocialService;

  @Autowired
  private Environment environment;

  @RequestMapping("/home.shtml")
  public String start(ModelMap modelMap, HttpServletRequest request,
      HttpServletResponse response) throws Exception {
    String owner = (String) request.getSession().getAttribute(
        LoginInterceptor.PERSON_SESSION_KEY);
    if (serviceConsumer.isOAuthUserConsentGranted(owner)) {
      try {
        getPersonInfo(modelMap, request, owner);
      } catch (Exception e) {
        // ok, apparently we were mistaken in believing we have an access key
        serviceConsumer.removeOAuthUserConsentGranted(owner);
      }
    }
    modelMap.addAttribute("guestStatus",
        request.getSession().getAttribute(LoginInterceptor.USER_STATUS));
    modelMap.addAttribute("licenseInfo",
        request.getSession().getAttribute(LoginInterceptor.LICENSE_INFO));
    modelMap.addAttribute("owner", owner);
    return "home";
  }

  private void getPersonInfo(ModelMap modelMap, HttpServletRequest request,
      String owner) {
    Person person = openSocialService.getPerson(owner);
    List<Group> teams = openSocialService.getTeams(owner);
    if (!CollectionUtils.isEmpty(teams)) {
      Group group = teams.get(0);
      List<Person> teamMembers = new ArrayList<Person>();
      try {
        teamMembers = openSocialService.getTeamMembers(owner,
            group.getId());
      } catch (Exception e) {
        // ignore, as grouper might be in bad modus
      }
      for (Person member : teamMembers) {
        if (!member.getId().equals(owner)) {
          try {
            Person detailsMember = openSocialService.getPerson(member.getId(),
                owner);
            group.put(detailsMember.getId(), detailsMember);
          } catch (Exception e) {
            // ignore, as grouper might be in bad modus
          }
        }
      }
    }
    String virtualOrganization = (String) request.getSession().getAttribute(
        VIRTUAL_ORGANIZATION);
    modelMap.addAttribute("person", person);
    modelMap.addAttribute("teams", teams);
    modelMap.addAttribute("members", null);
    modelMap.addAttribute("virtualOrganization", virtualOrganization);
  }

  @RequestMapping("/removeAccessTokens.shtml")
  public String removeAccessTokens(ModelMap modelMap,
      HttpServletRequest request, HttpServletResponse response)
      throws Exception {
    String owner = (String) request.getSession().getAttribute(
        LoginInterceptor.PERSON_SESSION_KEY);
    serviceConsumer.removeOAuthUserConsentGranted(owner);
    return "home";
  }

  @RequestMapping("/openSocialQueries.shtml")
  public String openSocialQueries(ModelMap modelMap,
      HttpServletRequest request, HttpServletResponse response)
      throws Exception {
    String owner = (String) request.getSession().getAttribute(
        LoginInterceptor.PERSON_SESSION_KEY);
    boolean oAuthUserConsentGranted = serviceConsumer
        .isOAuthUserConsentGranted(owner);
    modelMap.addAttribute("owner", owner);
    modelMap.addAttribute("granted", oAuthUserConsentGranted ? "Granted"
        : "Not Granted");
    return "queries";
  }

  @RequestMapping("/oauthCallBack.shtml")
  public void oauth(ModelMap modelMap, HttpServletRequest request,
      HttpServletResponse response) throws Exception {
    serviceConsumer.oauthCallBack(modelMap, request, response);
    response.sendRedirect("/testsp/home.shtml");
  }

  @RequestMapping(value = "/doUserConsentOAuth.shtml", params = "oauth")
  public String userConsent(ModelMap modelMap, HttpServletRequest request,
      HttpServletResponse response) throws Exception {
    String owner = (String) request.getSession().getAttribute(
        LoginInterceptor.PERSON_SESSION_KEY);
    String virtualOrganization = request.getParameter("virtualOrganization");
    if (StringUtils.hasText(virtualOrganization)) {
      request.getSession().setAttribute(VIRTUAL_ORGANIZATION,
          virtualOrganization);
    }
    serviceConsumer.provideOauthConsent(owner, "/home.shtml", request,
        response, virtualOrganization);
    return "home";
  }

  @RequestMapping(value = "/doOpenSocialQuery.shtml", params = "openSocialInformation")
  public String personInformation(ModelMap modelMap,
      HttpServletRequest request, HttpServletResponse response)
      throws Exception {
    String personId = request.getParameter("personId");
    String groupPersonId = request.getParameter("groupPersonId");
    String memberGroupId = request.getParameter("memberGroupId");
    String owner = (String) request.getSession().getAttribute(
        LoginInterceptor.PERSON_SESSION_KEY);
    Person person = null;
    List<Group> teams = null;
    List<Person> members = null;
    boolean oAuthUserConsentGranted = serviceConsumer
        .isOAuthUserConsentGranted(owner);
    if (oAuthUserConsentGranted) {
      if (StringUtils.hasText(personId)) {
        person = openSocialService.getPerson(personId, owner);
      }
      if (StringUtils.hasText(groupPersonId)) {
        teams = openSocialService.getTeams(groupPersonId, owner);
      }
      if (StringUtils.hasText(memberGroupId)) {
        if (StringUtils.hasText(personId)) {
          members = openSocialService.getTeamMembers(personId, owner,
              memberGroupId);
        } else {
          members = openSocialService.getTeamMembers(owner, owner,
              memberGroupId);
        }
      }

    }
    modelMap.addAttribute("owner", owner);
    modelMap.addAttribute("person", person);
    modelMap.addAttribute("teams", teams);
    modelMap.addAttribute("members", members);
    modelMap.addAttribute("granted", oAuthUserConsentGranted ? "Granted"
        : "Not Granted");
    return "queries";
  }

  @RequestMapping("/gadget.xml")
  public String gadgetXML(ModelMap modelMap, HttpServletRequest request,
      HttpServletResponse response) throws Exception {
    Provider provider = this.serviceConsumer.getProvider(request
        .getParameter("vo"));
    modelMap.addAttribute("accessUrl", provider.getAccessTokenUrl());
    modelMap.addAttribute("requestUrl", provider.getRequestTokenUrl());
    modelMap.addAttribute("authorizationUrl", provider.getAuthorizeUrl());
    String owner = (String) request.getSession().getAttribute(
        LoginInterceptor.PERSON_SESSION_KEY);
    modelMap.addAttribute("loggedInUser", owner);
    return "testsp-gadget";
  }

  @RequestMapping("/gadget-iframe.xml")
  public String start(ModelMap modelMap, HttpServletRequest request) {
    String gadgetBaseUrl = environment.getGadgetBaseUrl();
    if (!gadgetBaseUrl.endsWith("/")) {
      gadgetBaseUrl = (gadgetBaseUrl + "/");
    }
    modelMap.addAttribute("testSPURL", gadgetBaseUrl);
    return "testsp-gadget-iframe";
  }

}
