/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package nl.surfnet.coin.testsp.service;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.opensocial.Client;
import org.opensocial.auth.OAuth3LeggedScheme.Token;
import org.opensocial.providers.Provider;
import org.springframework.ui.ModelMap;

/**
 * Responsible for three-legged oauth
 * 
 */
public interface OAuthServiceConsumer {

  /**
   * The underlying OpenSocial implementations use Oauth in our case. Consent
   * (e.g. AccessTokens) are needed to query the OAuthProvider
   * 
   * @param personId
   *          the unique identifier for the person
   * @param request
   *          the request to perform redirecting if necessary
   * @throws Exception
   */
  void provideOauthConsent(String personId, String callback,
      HttpServletRequest request, HttpServletResponse response, String virtualOrganization)
      throws Exception;

  /**
   * Return a token for three-legged oauth
   * 
   * @param personId
   * @return the Token to use when communicating
   */
  Token getToken(String personId);

  /**
   * Get a OpenSocial client for two-legged oauth
   * 
   * @return Client for making request
   */
  Client getClient();

  /**
   * Get a OpenSocial client for two-legged oauth
   * 
   * @param token
   *          the accessToken
   * @return Client for making request
   */
  Client getClient(Token token);

  /**
   * @return true is there is a valid accessToken
   */
  boolean isOAuthUserConsentGranted(String personId);

  /**
   * Callback when the OAuth provider is returning the accessToken
   * 
   * @param modelMap
   *          Spring modelMap
   * @param request
   *          httpRequest
   * @param response
   *          httpResponse
   * @throws Exception
   */
  void oauthCallBack(ModelMap modelMap, HttpServletRequest request,
      HttpServletResponse response) throws Exception;

  /**
   * @param virtualOrganization
   * @return
   */
  Provider getProvider(String virtualOrganization);

  /**
   * @param owner
   */
  void removeOAuthUserConsentGranted(String owner);

}
