/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package nl.surfnet.coin.testsp.service;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nl.surfnet.coin.testsp.interceptor.LoginInterceptor;
import nl.surfnet.coin.testsp.util.Environment;

import org.opensocial.Client;
import org.opensocial.auth.AuthScheme;
import org.opensocial.auth.OAuth2LeggedScheme;
import org.opensocial.auth.OAuth3LeggedScheme;
import org.opensocial.auth.OAuth3LeggedScheme.Token;
import org.opensocial.providers.Provider;
import org.opensocial.providers.ShindigProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.ui.ModelMap;
import org.springframework.util.StringUtils;

/**
 *  
 *
 */
@Component("OAuthServiceConsumer")
public class OAuthServiceConsumerImpl implements OAuthServiceConsumer {

  private static final Map<String, Token> ACCESS_TOKENS = new HashMap<String, Token>();

  private static final Map<String, Token> REQUEST_TOKENS = new HashMap<String, Token>();

  //private static final String REQUEST_TOKEN = "REQUEST_TOKEN";

  @Autowired
  private Environment environment;

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.calendar.service.OAuthServiceConsumer#getToken(java.lang
   * .String)
   */
  @Override
  public Token getToken(String personId) {
    return ACCESS_TOKENS.get(personId);
  }

  /*
   * (non-Javadoc)
   * 
   * @see nl.surfnet.coin.calendar.service.OAuthServiceConsumer#getClient()
   */
  @Override
  public Client getClient() {
    Provider provider = getProvider(null);
    AuthScheme scheme = new OAuth2LeggedScheme(environment.getOauthKey(),
        environment.getOauthSecret(), LoginInterceptor.getLoggedInUser());
    return new Client(provider, scheme);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.calendar.service.OAuthServiceConsumer#getClient(org.opensocial
   * .auth.OAuth3LeggedScheme.Token)
   */
  @Override
  public Client getClient(Token token) {
    Provider provider = getProvider(null);

    OAuth3LeggedScheme scheme = new OAuth3LeggedScheme(provider,
        environment.getOauthKey(), environment.getOauthSecret());
    scheme.setAccessToken(token);
    return new Client(provider, scheme);
  }
 @Override
  public Provider getProvider(String virtualOrganization) {
    Provider provider = new ShindigProvider(true);

    String openSocialUrl = environment.getOpenSocialUrl();
    if (!openSocialUrl.endsWith("/")) {
      openSocialUrl = openSocialUrl + "/";
    }
    String voQuery = "";
    if (StringUtils.hasText(virtualOrganization)) {
      try {
        voQuery = "?vo=" + URLEncoder.encode(virtualOrganization, "UTF-8");
      } catch (UnsupportedEncodingException e) {
        //can't be helped
      }
    }

    provider.setRestEndpoint(openSocialUrl + "social/rest/");
    provider.setRpcEndpoint(null);
    provider.setVersion("0.9");

    provider.setAccessTokenUrl(openSocialUrl + "oauth/accessToken");
    provider.setAuthorizeUrl(openSocialUrl + "oauth/authorize");
    provider.setRequestTokenUrl(openSocialUrl + "oauth/requestToken"+voQuery);

    return provider;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.calendar.service.OAuthServiceConsumer#provideOauthConsent
   * (java.lang.String, java.lang.String, javax.servlet.http.HttpServletRequest,
   * javax.servlet.http.HttpServletResponse)
   */
  @Override
  public void provideOauthConsent(String personId, String callback,
      HttpServletRequest request, HttpServletResponse response , String virtualOrganization)
      throws Exception {
    Provider provider = getProvider(virtualOrganization);
    OAuth3LeggedScheme scheme = new OAuth3LeggedScheme(provider,
        environment.getOauthKey(), environment.getOauthSecret());

    String baseUrl = environment.getGadgetBaseUrl();
    if (!baseUrl.endsWith("/")) {
      baseUrl = baseUrl + "/";
    }
    callback = URLEncoder.encode(callback, "UTF-8");
    String authorizationUrl = scheme.getAuthorizationUrl(baseUrl
        + "oauthCallBack.shtml");
    REQUEST_TOKENS.put(personId, scheme.getRequestToken()) ;
    //request.getSession().setAttribute(REQUEST_TOKEN, scheme.getRequestToken());
    response.sendRedirect(authorizationUrl);

  }

  @Override
  public void oauthCallBack(ModelMap modelMap, HttpServletRequest request,
      HttpServletResponse response) throws Exception {
    Provider provider = getProvider( null);
    OAuth3LeggedScheme scheme = new OAuth3LeggedScheme(provider,
        environment.getOauthKey(), environment.getOauthSecret());

    String oAuthToken = request.getParameter("oauth_token");
    String oAuthVerifier = request.getParameter("oauth_verifier");
    String userId = request.getParameter("user_id");
    userId = URLDecoder.decode(userId, "UTF-8");
//    Token attribute = (Token) request.getSession().getAttribute(
//        REQUEST_TOKEN);
    Token attribute = REQUEST_TOKENS.get(userId);
    scheme.setRequestToken(attribute);
    scheme.requestAccessToken(oAuthToken, oAuthVerifier);
    Token accessToken = scheme.getAccessToken();
    String owner = (String) request.getSession().getAttribute(
        LoginInterceptor.PERSON_SESSION_KEY);
    ACCESS_TOKENS.put(owner, accessToken);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * nl.surfnet.coin.calendar.service.OAuthServiceConsumer#isOAuthConsentProvided
   * (java.lang.String)
   */
  @Override
  public boolean isOAuthUserConsentGranted(String personId) {
    return ACCESS_TOKENS.containsKey(personId);
  }

  /**
   * @param environment
   *          the environment to set
   */
  public void setEnvironment(Environment environment) {
    this.environment = environment;
  }

  /* (non-Javadoc)
   * @see nl.surfnet.coin.testsp.service.OAuthServiceConsumer#removeOAuthUserConsentGranted(java.lang.String)
   */
  @Override
  public void removeOAuthUserConsentGranted(String owner) {
    ACCESS_TOKENS.remove(owner);
    REQUEST_TOKENS.remove(owner);
  }

}
