/*
 * Copyright 2012 SURFnet bv, The Netherlands
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package nl.surfnet.coin.csa.service.impl;

import nl.surfnet.coin.csa.dao.CompoundServiceProviderDao;
import nl.surfnet.coin.csa.domain.Article;
import nl.surfnet.coin.csa.domain.CompoundServiceProvider;
import nl.surfnet.coin.csa.domain.IdentityProvider;
import nl.surfnet.coin.csa.domain.ServiceProvider;
import nl.surfnet.coin.csa.model.License;
import nl.surfnet.coin.csa.service.CrmService;
import nl.surfnet.coin.csa.service.ServiceProviderService;
import org.hibernate.HibernateException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.AbstractMap.SimpleEntry;
import java.util.*;

/**
 * Abstraction for the Compound Service Providers. This deals with persistence
 * and linking to Service Providers
 */
@Component
public class CompoundSPService {

  private Logger LOG = LoggerFactory.getLogger(CompoundSPService.class);

  @Resource
  private CompoundServiceProviderDao compoundServiceProviderDao;

  @Resource(name = "providerService")
  private ServiceProviderService serviceProviderService;

  @Resource
  private CrmService licensingService;

  public List<CompoundServiceProvider> getAllCSPs() {
    return getAllCSPs(0);
  }

  public List<CompoundServiceProvider> getAllCSPs(long callDelay) {
    List<ServiceProvider> allServiceProviders;
    if (callDelay > 0) {
      allServiceProviders = serviceProviderService.getAllServiceProvidersRateLimited(callDelay);
    } else {
      allServiceProviders = serviceProviderService.getAllServiceProviders(true);
    }
    return getCSPs(null, allServiceProviders);
  }

  public List<CompoundServiceProvider> getAllBareCSPs() {
    List<ServiceProvider> allServiceProviders = serviceProviderService.getAllServiceProviders(false);
    return getCSPs(null, allServiceProviders);
  }

  public List<CompoundServiceProvider> getCSPsByIdp(IdentityProvider identityProvider) {
    if (identityProvider == null) {
      return new ArrayList<CompoundServiceProvider>();
    }
    // Base: the list of all service providers for this IDP
    List<ServiceProvider> allServiceProviders = serviceProviderService.getAllServiceProviders(identityProvider.getId());
    return getCSPs(identityProvider, allServiceProviders);
  }

  private List<CompoundServiceProvider> getCSPs(IdentityProvider identityProvider, List<ServiceProvider> allServiceProviders) {
    // Reference data: all compound service providers
    List<CompoundServiceProvider> allBareCSPs = compoundServiceProviderDao.findAll();
    // Mapped by its SP entity ID
    Map<String, CompoundServiceProvider> mapByServiceProviderEntityId = mapByServiceProviderEntityId(allBareCSPs);

    // Build a list of CSPs. Create new ones for SPs that have no CSP yet.
    List<CompoundServiceProvider> all = new ArrayList<CompoundServiceProvider>();
    for (ServiceProvider sp : allServiceProviders) {

      CompoundServiceProvider csp;
      if (mapByServiceProviderEntityId.containsKey(sp.getId())) {
        csp = mapByServiceProviderEntityId.get(sp.getId());
        csp = compound(identityProvider, sp, csp);
      } else {
        LOG.debug("No CompoundServiceProvider yet for SP with id {}, will create a new one.", sp.getId());
        csp = createCompoundServiceProvider(identityProvider, sp);
      }
      all.add(csp);
    }
    return all;
  }

  private CompoundServiceProvider compound(IdentityProvider identityProvider, ServiceProvider sp, CompoundServiceProvider csp) {
    csp.setServiceProvider(sp);
    Article article = getArticleForSp(sp);
    csp.setArticle(article);
    if (identityProvider != null) {
      csp.setLicenses(getLicensesForIdpAndArticle(identityProvider, article));
    }
    return csp;
  }

  /**
   * Create a CSP for the given SP.
   *
   * @param sp the SP
   * @return the created (and persisted) CSP
   */
  private CompoundServiceProvider createCompoundServiceProvider(IdentityProvider idp, ServiceProvider sp) {
    Article article = getArticleForSp(sp);
    CompoundServiceProvider csp = CompoundServiceProvider.builder(sp, article);
    if (idp != null) {
      csp.setLicenses(getLicensesForIdpAndArticle(idp, article));
    }
    try {
      compoundServiceProviderDao.saveOrUpdate(csp);
    } catch (RuntimeException e) {
      if (e instanceof HibernateException || e instanceof DataAccessException) {
        //let's give the database another try, otherwise rethrow
        CompoundServiceProvider cspFromDb = compoundServiceProviderDao.findByEntityId(sp.getId());
        if (cspFromDb != null) {
          csp = compound(idp, sp, cspFromDb);
        } else {
          throw e;
        }
      }
    }
    return csp;
  }

  private Map<String, CompoundServiceProvider> mapByServiceProviderEntityId(List<CompoundServiceProvider> allCSPs) {
    Map<String, CompoundServiceProvider> map = new HashMap<String, CompoundServiceProvider>();
    for (CompoundServiceProvider csp : allCSPs) {
      map.put(csp.getServiceProviderEntityId(), csp);
    }
    return map;
  }

  /**
   * Get a CSP by its ID, for the given IDP.
   *
   * @param idp          the IDP
   * @param compoundSpId long
   * @return
   */
  public CompoundServiceProvider getCSPById(IdentityProvider idp, long compoundSpId) {
    CompoundServiceProvider csp = compoundServiceProviderDao.findById(compoundSpId);
    if (csp == null) {
      LOG.debug("Cannot find CSP by id {}, will return null", compoundSpId);
      return null;
    }
    ServiceProvider sp = serviceProviderService.getServiceProvider(csp.getServiceProviderEntityId(), idp.getId());
    if (sp == null) {
      LOG.info("Cannot get serviceProvider by known entity id: {}, cannot enrich CSP with SP information.",
              csp.getServiceProviderEntityId());
      return csp;
    }
    csp.setServiceProvider(sp);
    Article article = getArticleForSp(sp);
    csp.setArticle(article);
    csp.setLicenses(getLicensesForIdpAndArticle(idp, article));
    return csp;
  }

  /**
   * Get a CSP by its ServiceProvider
   *
   * @param serviceProviderEntityId the ServiceProvider
   * @return
   */
  public CompoundServiceProvider getCSPByServiceProviderEntityId(String serviceProviderEntityId) {

    ServiceProvider serviceProvider = serviceProviderService.getServiceProvider(serviceProviderEntityId);
    Assert.notNull(serviceProvider, "No such SP with entityId: " + serviceProviderEntityId);
    return getCSPByServiceProvider(serviceProvider);
  }

  /**
   * Get a CSP by its ServiceProvider
   *
   * @param serviceProvider the ServiceProvider
   * @return
   */
  public CompoundServiceProvider getCSPByServiceProvider(ServiceProvider serviceProvider) {

    Assert.notNull(serviceProvider, "ServiceProvider may not be null");
    CompoundServiceProvider compoundServiceProvider = compoundServiceProviderDao.findByEntityId(serviceProvider.getId());
    if (compoundServiceProvider == null) {
      LOG.debug("No compound Service Provider for SP '{}' yet. Will init one and persist.", serviceProvider.getId());
      compoundServiceProvider = CompoundServiceProvider.builder(serviceProvider, getArticleForSp(serviceProvider));
      compoundServiceProviderDao.saveOrUpdate(compoundServiceProvider);
      LOG.debug("Persisted a CompoundServiceProvider with id {}");
    } else {
      compoundServiceProvider.setServiceProvider(serviceProvider);
      compoundServiceProvider.setArticle(getArticleForSp(serviceProvider));
      compoundServiceProvider.updateTransientOriginFields();
    }
    return compoundServiceProvider;
  }

  private Article getArticleForSp(ServiceProvider sp) {
    Assert.notNull(sp);

    List<String> allSpsIds = new ArrayList<String>();
    allSpsIds.add(sp.getId());
    List<Article> articles = licensingService.getArticlesForServiceProviders(allSpsIds);

    for (Article article : articles) {
      if (article.getServiceProviderEntityId().equals(sp.getId())) {
        return article;
      }
    }
    return null;
  }

  private List<License> getLicensesForIdpAndArticle(IdentityProvider idp, Article article) {
    Assert.notNull(idp);
    if (article != null) {
      List<License> licenses = licensingService.getLicensesForIdpAndSp(idp, article.getLmngIdentifier());
      return licenses;
    }

    return null;
  }

}
